<?php

namespace Tests\Unit;

use App\Models\Contact;
use App\Models\Tenant;
use App\Models\User;
use App\Models\Workflow;
use App\Models\WorkflowEnrollment;
use App\Services\WorkflowService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class WorkflowServiceTest extends TestCase
{
    use RefreshDatabase;

    protected WorkflowService $service;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();

        $this->service = app(WorkflowService::class);
        $this->tenant = Tenant::factory()->create();
    }

    public function test_creates_workflow_with_steps(): void
    {
        $workflow = $this->service->create($this->tenant->id, [
            'name' => 'Welcome Sequence',
            'trigger_type' => 'contact_created',
            'steps' => [
                ['type' => 'send_email', 'config' => ['template_id' => 1], 'delay_minutes' => 0],
                ['type' => 'wait', 'config' => [], 'delay_minutes' => 1440],
                ['type' => 'add_tag', 'config' => ['tag_id' => 1], 'delay_minutes' => 0],
            ],
        ]);

        $this->assertCount(3, $workflow->steps);
        $this->assertEquals('send_email', $workflow->steps[0]->type);
    }

    public function test_enrolls_contact_in_workflow(): void
    {
        $workflow = Workflow::factory()->create([
            'tenant_id' => $this->tenant->id,
            'is_active' => true,
        ]);

        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);

        $enrollment = $this->service->enroll($workflow, $contact);

        $this->assertEquals('active', $enrollment->status);
        $this->assertEquals(1, $enrollment->current_step);
    }

    public function test_unenrolls_contact(): void
    {
        $workflow = Workflow::factory()->create(['tenant_id' => $this->tenant->id]);
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
        
        $enrollment = WorkflowEnrollment::factory()->create([
            'tenant_id' => $this->tenant->id,
            'workflow_id' => $workflow->id,
            'contact_id' => $contact->id,
            'status' => 'active',
        ]);

        $this->service->unenroll($enrollment, 'manual');

        $this->assertEquals('unenrolled', $enrollment->fresh()->status);
    }

    public function test_checks_trigger_conditions(): void
    {
        $workflow = Workflow::factory()->create([
            'tenant_id' => $this->tenant->id,
            'trigger_type' => 'contact_created',
            'trigger_conditions' => [
                ['field' => 'source', 'operator' => 'equals', 'value' => 'website'],
            ],
            'is_active' => true,
        ]);

        $matchingContact = Contact::factory()->create([
            'tenant_id' => $this->tenant->id,
            'source' => 'website',
        ]);

        $nonMatchingContact = Contact::factory()->create([
            'tenant_id' => $this->tenant->id,
            'source' => 'referral',
        ]);

        $this->service->triggerByEvent($this->tenant->id, 'contact_created', $matchingContact);
        $this->service->triggerByEvent($this->tenant->id, 'contact_created', $nonMatchingContact);

        $this->assertTrue(WorkflowEnrollment::where('contact_id', $matchingContact->id)->exists());
        $this->assertFalse(WorkflowEnrollment::where('contact_id', $nonMatchingContact->id)->exists());
    }
}
