<?php

namespace Tests\Unit;

use App\Models\Project;
use App\Models\Task;
use App\Models\TaskStatus;
use App\Models\Tenant;
use App\Models\User;
use App\Services\TaskService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class TaskServiceTest extends TestCase
{
    use RefreshDatabase;

    protected TaskService $service;
    protected Tenant $tenant;
    protected User $user;
    protected Project $project;
    protected TaskStatus $todoStatus;
    protected TaskStatus $doneStatus;

    protected function setUp(): void
    {
        parent::setUp();

        $this->service = app(TaskService::class);
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->project = Project::factory()->create(['tenant_id' => $this->tenant->id]);
        
        $this->todoStatus = TaskStatus::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
            'name' => 'To Do',
            'is_default' => true,
        ]);
        
        $this->doneStatus = TaskStatus::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
            'name' => 'Done',
            'is_completed' => true,
        ]);
    }

    public function test_creates_task_with_default_status(): void
    {
        $data = [
            'project_id' => $this->project->id,
            'title' => 'New Task',
            'description' => 'Task description',
        ];

        $task = $this->service->create($this->tenant->id, $data, $this->user->id);

        $this->assertInstanceOf(Task::class, $task);
        $this->assertEquals($this->todoStatus->id, $task->status_id);
        $this->assertEquals('New Task', $task->title);
    }

    public function test_creates_task_with_assignees(): void
    {
        $assignee = User::factory()->create(['tenant_id' => $this->tenant->id]);

        $data = [
            'project_id' => $this->project->id,
            'title' => 'Assigned Task',
            'assignees' => [$assignee->id],
        ];

        $task = $this->service->create($this->tenant->id, $data, $this->user->id);

        $this->assertEquals(1, $task->assignees->count());
        $this->assertEquals($assignee->id, $task->assignees->first()->user_id);
    }

    public function test_completes_task(): void
    {
        $task = Task::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
            'status_id' => $this->todoStatus->id,
        ]);

        $this->service->complete($task, $this->user->id);

        $task->refresh();
        $this->assertEquals($this->doneStatus->id, $task->status_id);
        $this->assertNotNull($task->completed_at);
        $this->assertEquals($this->user->id, $task->completed_by);
    }

    public function test_reopens_task(): void
    {
        $task = Task::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
            'status_id' => $this->doneStatus->id,
            'completed_at' => now(),
            'completed_by' => $this->user->id,
        ]);

        $this->service->reopen($task);

        $task->refresh();
        $this->assertEquals($this->todoStatus->id, $task->status_id);
        $this->assertNull($task->completed_at);
    }

    public function test_creates_subtask(): void
    {
        $parentTask = Task::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
        ]);

        $data = [
            'project_id' => $this->project->id,
            'parent_id' => $parentTask->id,
            'title' => 'Subtask',
        ];

        $subtask = $this->service->create($this->tenant->id, $data, $this->user->id);

        $this->assertEquals($parentTask->id, $subtask->parent_id);
        $this->assertEquals(1, $parentTask->subtasks->count());
    }
}
