<?php

namespace Tests\Unit;

use App\Models\Contact;
use App\Models\Expense;
use App\Models\Invoice;
use App\Models\Opportunity;
use App\Models\Pipeline;
use App\Models\PipelineStage;
use App\Models\Tenant;
use App\Services\ReportService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ReportServiceTest extends TestCase
{
    use RefreshDatabase;

    protected ReportService $service;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();

        $this->service = app(ReportService::class);
        $this->tenant = Tenant::factory()->create();
    }

    public function test_calculates_revenue_for_period(): void
    {
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
        
        Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $contact->id,
            'status' => 'paid',
            'total' => 1000.000,
            'invoice_date' => now(),
        ]);
        Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $contact->id,
            'status' => 'paid',
            'total' => 500.000,
            'invoice_date' => now(),
        ]);

        $revenue = $this->service->getRevenueForPeriod(
            $this->tenant->id,
            now()->startOfMonth()->toDateString(),
            now()->endOfMonth()->toDateString()
        );

        $this->assertEquals(1500.000, $revenue);
    }

    public function test_calculates_expenses_for_period(): void
    {
        Expense::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'approved',
            'amount' => 250.000,
            'expense_date' => now(),
        ]);
        Expense::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'approved',
            'amount' => 150.000,
            'expense_date' => now(),
        ]);

        $expenses = $this->service->getExpensesForPeriod(
            $this->tenant->id,
            now()->startOfMonth()->toDateString(),
            now()->endOfMonth()->toDateString()
        );

        $this->assertEquals(400.000, $expenses);
    }

    public function test_calculates_won_opportunities(): void
    {
        $pipeline = Pipeline::factory()->create(['tenant_id' => $this->tenant->id]);
        $stage = PipelineStage::factory()->create(['pipeline_id' => $pipeline->id]);
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);

        Opportunity::factory()->create([
            'tenant_id' => $this->tenant->id,
            'pipeline_id' => $pipeline->id,
            'stage_id' => $stage->id,
            'contact_id' => $contact->id,
            'status' => 'won',
            'value' => 5000.000,
            'closed_at' => now(),
        ]);

        $wonValue = $this->service->getWonOpportunitiesForPeriod(
            $this->tenant->id,
            now()->startOfMonth()->toDateString(),
            now()->endOfMonth()->toDateString()
        );

        $this->assertEquals(5000.000, $wonValue);
    }

    public function test_generates_profit_and_loss(): void
    {
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
        
        Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $contact->id,
            'status' => 'paid',
            'total' => 2000.000,
            'invoice_date' => now(),
        ]);
        
        Expense::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'approved',
            'amount' => 500.000,
            'expense_date' => now(),
        ]);

        $report = $this->service->generateProfitAndLoss(
            $this->tenant->id,
            now()->startOfMonth()->toDateString(),
            now()->endOfMonth()->toDateString()
        );

        $this->assertEquals(2000.000, $report['revenue']['total']);
        $this->assertGreaterThan(0, $report['net_income']);
    }
}
