<?php

namespace Tests\Unit;

use App\Models\Contact;
use App\Models\Opportunity;
use App\Models\Pipeline;
use App\Models\PipelineStage;
use App\Models\Tenant;
use App\Models\User;
use App\Services\OpportunityService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class OpportunityServiceTest extends TestCase
{
    use RefreshDatabase;

    protected OpportunityService $service;
    protected Tenant $tenant;
    protected User $user;
    protected Pipeline $pipeline;
    protected PipelineStage $stage;
    protected Contact $contact;

    protected function setUp(): void
    {
        parent::setUp();

        $this->service = app(OpportunityService::class);
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->pipeline = Pipeline::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->stage = PipelineStage::factory()->create(['pipeline_id' => $this->pipeline->id]);
        $this->contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_creates_opportunity(): void
    {
        $opportunity = $this->service->create($this->tenant->id, [
            'name' => 'Test Deal',
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
            'value' => 5000,
        ], $this->user->id);

        $this->assertInstanceOf(Opportunity::class, $opportunity);
        $this->assertEquals('open', $opportunity->status);
    }

    public function test_updates_stage(): void
    {
        $opportunity = Opportunity::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
        ]);

        $newStage = PipelineStage::factory()->create(['pipeline_id' => $this->pipeline->id]);

        $updated = $this->service->updateStage($opportunity, $newStage->id, $this->user->id);

        $this->assertEquals($newStage->id, $updated->stage_id);
    }

    public function test_marks_won_and_updates_contact(): void
    {
        $opportunity = Opportunity::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
            'status' => 'open',
        ]);

        $this->service->markWon($opportunity, $this->user->id);

        $this->assertEquals('won', $opportunity->fresh()->status);
        $this->assertEquals('customer', $this->contact->fresh()->lead_status);
    }

    public function test_marks_lost_with_reason(): void
    {
        $opportunity = Opportunity::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
            'status' => 'open',
        ]);

        $this->service->markLost($opportunity, $this->user->id, 'Too expensive');

        $this->assertEquals('lost', $opportunity->fresh()->status);
        $this->assertEquals('Too expensive', $opportunity->fresh()->lost_reason);
    }
}
