<?php

namespace Tests\Unit;

use App\Models\Contact;
use App\Models\Invoice;
use App\Models\Tenant;
use App\Models\User;
use App\Services\InvoiceService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class InvoiceServiceTest extends TestCase
{
    use RefreshDatabase;

    protected InvoiceService $service;
    protected Tenant $tenant;
    protected User $user;
    protected Contact $contact;

    protected function setUp(): void
    {
        parent::setUp();

        $this->service = app(InvoiceService::class);
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_creates_invoice_with_items(): void
    {
        $data = [
            'contact_id' => $this->contact->id,
            'invoice_date' => now()->toDateString(),
            'due_date' => now()->addDays(30)->toDateString(),
            'items' => [
                ['description' => 'Service A', 'quantity' => 1, 'unit_price' => 500.000],
                ['description' => 'Service B', 'quantity' => 2, 'unit_price' => 250.000],
            ],
        ];

        $invoice = $this->service->create($this->tenant->id, $data, $this->user->id);

        $this->assertInstanceOf(Invoice::class, $invoice);
        $this->assertEquals(1000.000, $invoice->subtotal);
        $this->assertEquals(2, $invoice->items->count());
    }

    public function test_calculates_tax_correctly(): void
    {
        $data = [
            'contact_id' => $this->contact->id,
            'invoice_date' => now()->toDateString(),
            'due_date' => now()->addDays(30)->toDateString(),
            'items' => [
                ['description' => 'Service', 'quantity' => 1, 'unit_price' => 1000.000, 'tax_rate' => 10],
            ],
        ];

        $invoice = $this->service->create($this->tenant->id, $data, $this->user->id);

        $this->assertEquals(100.000, $invoice->tax_amount);
        $this->assertEquals(1100.000, $invoice->total);
    }

    public function test_records_partial_payment(): void
    {
        $invoice = Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'total' => 1000.000,
            'status' => 'sent',
        ]);

        $this->service->recordPayment($invoice, [
            'amount' => 400.000,
            'payment_date' => now()->toDateString(),
            'payment_method' => 'bank_transfer',
        ], $this->user->id);

        $invoice->refresh();
        $this->assertEquals(400.000, $invoice->amount_paid);
        $this->assertEquals('partial', $invoice->status);
    }

    public function test_records_full_payment(): void
    {
        $invoice = Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'total' => 1000.000,
            'status' => 'sent',
        ]);

        $this->service->recordPayment($invoice, [
            'amount' => 1000.000,
            'payment_date' => now()->toDateString(),
            'payment_method' => 'bank_transfer',
        ], $this->user->id);

        $invoice->refresh();
        $this->assertEquals('paid', $invoice->status);
    }

    public function test_generates_unique_invoice_number(): void
    {
        $invoice1 = $this->service->create($this->tenant->id, [
            'contact_id' => $this->contact->id,
            'invoice_date' => now()->toDateString(),
            'due_date' => now()->addDays(30)->toDateString(),
            'items' => [['description' => 'Test', 'quantity' => 1, 'unit_price' => 100]],
        ], $this->user->id);

        $invoice2 = $this->service->create($this->tenant->id, [
            'contact_id' => $this->contact->id,
            'invoice_date' => now()->toDateString(),
            'due_date' => now()->addDays(30)->toDateString(),
            'items' => [['description' => 'Test', 'quantity' => 1, 'unit_price' => 100]],
        ], $this->user->id);

        $this->assertNotEquals($invoice1->invoice_number, $invoice2->invoice_number);
    }
}
