<?php

namespace Tests\Feature;

use App\Models\Tenant;
use App\Models\User;
use App\Models\Workflow;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class WorkflowTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();

        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_workflows(): void
    {
        Workflow::factory()->count(2)->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)->getJson('/api/workflows');

        $response->assertOk()->assertJsonCount(2, 'data');
    }

    public function test_can_create_workflow(): void
    {
        $data = [
            'name' => 'Welcome Workflow',
            'description' => 'Send welcome email to new contacts',
            'trigger' => [
                'type' => 'contact_created',
                'conditions' => [
                    ['field' => 'source', 'operator' => 'equals', 'value' => 'website'],
                ],
            ],
            'steps' => [
                [
                    'type' => 'send_email',
                    'config' => ['template_id' => 1],
                ],
                [
                    'type' => 'wait',
                    'delay' => ['value' => 3, 'unit' => 'days'],
                    'config' => [],
                ],
                [
                    'type' => 'add_tag',
                    'config' => ['tag_ids' => [1]],
                ],
            ],
        ];

        $response = $this->actingAs($this->user)->postJson('/api/workflows', $data);

        $response->assertCreated()
            ->assertJsonPath('data.name', 'Welcome Workflow')
            ->assertJsonPath('data.is_active', false);

        $this->assertEquals(3, $response->json('data.steps_count') ?? count($response->json('data.steps')));
    }

    public function test_can_activate_workflow(): void
    {
        $workflow = Workflow::factory()->create([
            'tenant_id' => $this->tenant->id,
            'is_active' => false,
        ]);

        $workflow->steps()->create([
            'tenant_id' => $this->tenant->id,
            'type' => 'send_email',
            'config' => ['template_id' => 1],
            'position' => 1,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/workflows/{$workflow->id}/activate");

        $response->assertOk();
        $this->assertTrue($workflow->fresh()->is_active);
    }

    public function test_cannot_activate_workflow_without_steps(): void
    {
        $workflow = Workflow::factory()->create([
            'tenant_id' => $this->tenant->id,
            'is_active' => false,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/workflows/{$workflow->id}/activate");

        $response->assertStatus(422);
    }
}
