<?php

namespace Tests\Feature;

use App\Models\Contact;
use App\Models\Tag;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class TagApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_tags(): void
    {
        Tag::factory()->count(5)->create(['tenant_id' => $this->tenant->id]);
        $response = $this->actingAs($this->user)->getJson('/api/tags');
        $response->assertOk()->assertJsonCount(5, 'data');
    }

    public function test_can_create_tag(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/tags', [
            'name' => 'VIP',
            'color' => '#FF0000',
        ]);
        $response->assertCreated()->assertJsonPath('data.name', 'VIP');
    }

    public function test_can_tag_contact(): void
    {
        $tag = Tag::factory()->create(['tenant_id' => $this->tenant->id]);
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/contacts/{$contact->id}/tags", [
                'tag_ids' => [$tag->id],
            ]);

        $response->assertOk();
        $this->assertTrue($contact->tags->contains($tag));
    }

    public function test_can_untag_contact(): void
    {
        $tag = Tag::factory()->create(['tenant_id' => $this->tenant->id]);
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
        $contact->tags()->attach($tag);

        $response = $this->actingAs($this->user)
            ->deleteJson("/api/contacts/{$contact->id}/tags/{$tag->id}");

        $response->assertOk();
        $this->assertFalse($contact->fresh()->tags->contains($tag));
    }

    public function test_deleting_tag_removes_from_contacts(): void
    {
        $tag = Tag::factory()->create(['tenant_id' => $this->tenant->id]);
        $contacts = Contact::factory()->count(3)->create(['tenant_id' => $this->tenant->id]);
        foreach ($contacts as $contact) {
            $contact->tags()->attach($tag);
        }

        $this->actingAs($this->user)->deleteJson("/api/tags/{$tag->id}");

        foreach ($contacts as $contact) {
            $this->assertFalse($contact->fresh()->tags->contains($tag));
        }
    }
}
