<?php

namespace Tests\Feature;

use App\Models\Contact;
use App\Models\Segment;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class SegmentApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_segments(): void
    {
        Segment::factory()->count(3)->create(['tenant_id' => $this->tenant->id]);
        $response = $this->actingAs($this->user)->getJson('/api/segments');
        $response->assertOk()->assertJsonCount(3, 'data');
    }

    public function test_can_create_static_segment(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/segments', [
            'name' => 'VIP Customers',
            'description' => 'High-value customers',
            'is_dynamic' => false,
        ]);
        $response->assertCreated()->assertJsonPath('data.name', 'VIP Customers');
    }

    public function test_can_create_dynamic_segment(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/segments', [
            'name' => 'Bahrain Leads',
            'is_dynamic' => true,
            'conditions' => [
                ['field' => 'country', 'operator' => 'equals', 'value' => 'BH'],
            ],
        ]);
        $response->assertCreated()->assertJsonPath('data.is_dynamic', true);
    }

    public function test_can_add_contacts_to_segment(): void
    {
        $segment = Segment::factory()->create(['tenant_id' => $this->tenant->id, 'is_dynamic' => false]);
        $contacts = Contact::factory()->count(3)->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/segments/{$segment->id}/contacts", [
                'contact_ids' => $contacts->pluck('id')->toArray(),
            ]);

        $response->assertOk();
        $this->assertEquals(3, $segment->contacts()->count());
    }

    public function test_can_refresh_dynamic_segment(): void
    {
        Contact::factory()->count(5)->create(['tenant_id' => $this->tenant->id, 'country' => 'BH']);
        Contact::factory()->count(3)->create(['tenant_id' => $this->tenant->id, 'country' => 'SA']);

        $segment = Segment::factory()->create([
            'tenant_id' => $this->tenant->id,
            'is_dynamic' => true,
            'conditions' => [['field' => 'country', 'operator' => 'equals', 'value' => 'BH']],
        ]);

        $response = $this->actingAs($this->user)->postJson("/api/segments/{$segment->id}/refresh");
        $response->assertOk();
    }
}
