<?php

namespace Tests\Feature;

use App\Models\Contact;
use App\Models\Opportunity;
use App\Models\Pipeline;
use App\Models\PipelineStage;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class OpportunityApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Pipeline $pipeline;
    protected PipelineStage $stage;
    protected Contact $contact;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->pipeline = Pipeline::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->stage = PipelineStage::factory()->create(['pipeline_id' => $this->pipeline->id]);
        $this->contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_create_opportunity(): void
    {
        $data = [
            'name' => 'Company Formation Deal',
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
            'value' => 2500.000,
        ];

        $response = $this->actingAs($this->user)->postJson('/api/opportunities', $data);

        $response->assertCreated()
            ->assertJsonPath('data.name', 'Company Formation Deal')
            ->assertJsonPath('data.status', 'open');
    }

    public function test_can_update_stage(): void
    {
        $opportunity = Opportunity::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
        ]);

        $newStage = PipelineStage::factory()->create(['pipeline_id' => $this->pipeline->id]);

        $response = $this->actingAs($this->user)
            ->putJson("/api/opportunities/{$opportunity->id}/stage", [
                'stage_id' => $newStage->id,
            ]);

        $response->assertOk();
        $this->assertEquals($newStage->id, $opportunity->fresh()->stage_id);
    }

    public function test_can_mark_won(): void
    {
        $opportunity = Opportunity::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
            'status' => 'open',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/opportunities/{$opportunity->id}/won");

        $response->assertOk();
        $this->assertEquals('won', $opportunity->fresh()->status);
        $this->assertEquals(100, $opportunity->fresh()->probability);
    }

    public function test_can_mark_lost(): void
    {
        $opportunity = Opportunity::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
            'status' => 'open',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/opportunities/{$opportunity->id}/lost", [
                'reason' => 'Budget constraints',
            ]);

        $response->assertOk();
        $this->assertEquals('lost', $opportunity->fresh()->status);
        $this->assertEquals('Budget constraints', $opportunity->fresh()->lost_reason);
    }

    public function test_calculates_weighted_value(): void
    {
        Opportunity::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'pipeline_id' => $this->pipeline->id,
            'stage_id' => $this->stage->id,
            'value' => 10000,
            'probability' => 50,
            'status' => 'open',
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/opportunities/stats');

        $response->assertOk()
            ->assertJsonPath('data.weighted_value', 5000);
    }
}
