<?php

namespace Tests\Feature;

use App\Models\Form;
use App\Models\FormSubmission;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class FormApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_create_form(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/forms', [
            'name' => 'Contact Form',
            'description' => 'Website contact form',
            'fields' => [
                ['label' => 'Name', 'name' => 'name', 'type' => 'text', 'required' => true],
                ['label' => 'Email', 'name' => 'email', 'type' => 'email', 'required' => true],
                ['label' => 'Message', 'name' => 'message', 'type' => 'textarea', 'required' => false],
            ],
        ]);

        $response->assertCreated()
            ->assertJsonPath('data.name', 'Contact Form')
            ->assertJsonStructure(['data' => ['id', 'slug', 'fields']]);
    }

    public function test_can_submit_form(): void
    {
        $form = Form::factory()->create([
            'tenant_id' => $this->tenant->id,
            'is_active' => true,
            'fields' => [
                ['label' => 'Name', 'name' => 'name', 'type' => 'text', 'required' => true],
                ['label' => 'Email', 'name' => 'email', 'type' => 'email', 'required' => true],
            ],
        ]);

        $response = $this->postJson("/api/forms/{$form->slug}/submit", [
            'name' => 'John Doe',
            'email' => 'john@example.com',
        ]);

        $response->assertOk()
            ->assertJsonStructure(['message']);

        $this->assertDatabaseHas('form_submissions', ['form_id' => $form->id]);
    }

    public function test_form_submission_creates_contact(): void
    {
        $form = Form::factory()->create([
            'tenant_id' => $this->tenant->id,
            'is_active' => true,
            'create_contact' => true,
            'fields' => [
                ['label' => 'Name', 'name' => 'first_name', 'type' => 'text', 'required' => true],
                ['label' => 'Email', 'name' => 'email', 'type' => 'email', 'required' => true],
            ],
        ]);

        $this->postJson("/api/forms/{$form->slug}/submit", [
            'first_name' => 'Jane',
            'email' => 'jane@example.com',
        ]);

        $this->assertDatabaseHas('contacts', [
            'tenant_id' => $this->tenant->id,
            'email' => 'jane@example.com',
        ]);
    }

    public function test_cannot_submit_inactive_form(): void
    {
        $form = Form::factory()->create([
            'tenant_id' => $this->tenant->id,
            'is_active' => false,
        ]);

        $response = $this->postJson("/api/forms/{$form->slug}/submit", [
            'name' => 'Test',
        ]);

        $response->assertNotFound();
    }

    public function test_can_list_form_submissions(): void
    {
        $form = Form::factory()->create(['tenant_id' => $this->tenant->id]);
        FormSubmission::factory()->count(5)->create(['form_id' => $form->id]);

        $response = $this->actingAs($this->user)->getJson("/api/forms/{$form->id}/submissions");

        $response->assertOk()
            ->assertJsonCount(5, 'data');
    }
}
