<?php

namespace Tests\Feature;

use App\Models\Expense;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ExpenseApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_expenses(): void
    {
        Expense::factory()->count(5)->create([
            'tenant_id' => $this->tenant->id,
            'created_by' => $this->user->id,
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/expenses');

        $response->assertOk()
            ->assertJsonCount(5, 'data');
    }

    public function test_can_create_expense(): void
    {
        $data = [
            'expense_date' => now()->toDateString(),
            'description' => 'Business lunch',
            'amount' => 25.500,
            'category' => 'meals',
        ];

        $response = $this->actingAs($this->user)->postJson('/api/expenses', $data);

        $response->assertCreated()
            ->assertJsonPath('data.description', 'Business lunch')
            ->assertJsonPath('data.status', 'pending');
    }

    public function test_can_approve_expense(): void
    {
        $expense = Expense::factory()->create([
            'tenant_id' => $this->tenant->id,
            'created_by' => $this->user->id,
            'status' => 'pending',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/expenses/{$expense->id}/approve");

        $response->assertOk();
        $this->assertEquals('approved', $expense->fresh()->status);
    }

    public function test_can_reject_expense(): void
    {
        $expense = Expense::factory()->create([
            'tenant_id' => $this->tenant->id,
            'created_by' => $this->user->id,
            'status' => 'pending',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/expenses/{$expense->id}/reject", [
                'reason' => 'Missing receipt',
            ]);

        $response->assertOk();
        $this->assertEquals('rejected', $expense->fresh()->status);
    }

    public function test_filters_expenses_by_category(): void
    {
        Expense::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'created_by' => $this->user->id,
            'category' => 'travel',
        ]);
        Expense::factory()->count(2)->create([
            'tenant_id' => $this->tenant->id,
            'created_by' => $this->user->id,
            'category' => 'meals',
        ]);

        $response = $this->actingAs($this->user)
            ->getJson('/api/expenses?category=travel');

        $response->assertOk()
            ->assertJsonCount(3, 'data');
    }
}
