<?php

namespace Tests\Feature;

use App\Models\Contact;
use App\Models\Tag;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ContactApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_contacts(): void
    {
        Contact::factory()->count(10)->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)->getJson('/api/contacts');

        $response->assertOk()
            ->assertJsonCount(10, 'data');
    }

    public function test_can_search_contacts(): void
    {
        Contact::factory()->create([
            'tenant_id' => $this->tenant->id,
            'first_name' => 'John',
            'last_name' => 'Doe',
        ]);
        Contact::factory()->create([
            'tenant_id' => $this->tenant->id,
            'first_name' => 'Jane',
            'last_name' => 'Smith',
        ]);

        $response = $this->actingAs($this->user)
            ->getJson('/api/contacts?search=John');

        $response->assertOk()
            ->assertJsonCount(1, 'data')
            ->assertJsonPath('data.0.first_name', 'John');
    }

    public function test_can_create_contact(): void
    {
        $data = [
            'first_name' => 'Ahmed',
            'last_name' => 'Al-Bahraini',
            'email' => 'ahmed@example.com',
            'company_name' => 'Setup in Bahrain',
            'lead_status' => 'new',
        ];

        $response = $this->actingAs($this->user)->postJson('/api/contacts', $data);

        $response->assertCreated()
            ->assertJsonPath('data.first_name', 'Ahmed')
            ->assertJsonPath('data.company_name', 'Setup in Bahrain');
    }

    public function test_can_add_tags_to_contact(): void
    {
        $contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
        $tag = Tag::factory()->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/contacts/{$contact->id}/tags", [
                'tag_ids' => [$tag->id],
            ]);

        $response->assertOk();
        $this->assertTrue($contact->fresh()->tags->contains($tag));
    }

    public function test_can_filter_contacts_by_status(): void
    {
        Contact::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'lead_status' => 'qualified',
        ]);
        Contact::factory()->count(2)->create([
            'tenant_id' => $this->tenant->id,
            'lead_status' => 'new',
        ]);

        $response = $this->actingAs($this->user)
            ->getJson('/api/contacts?lead_status=qualified');

        $response->assertOk()
            ->assertJsonCount(3, 'data');
    }

    public function test_validates_required_fields(): void
    {
        $response = $this->actingAs($this->user)
            ->postJson('/api/contacts', []);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['first_name', 'email']);
    }

    public function test_validates_unique_email(): void
    {
        Contact::factory()->create([
            'tenant_id' => $this->tenant->id,
            'email' => 'existing@example.com',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson('/api/contacts', [
                'first_name' => 'Test',
                'last_name' => 'User',
                'email' => 'existing@example.com',
            ]);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['email']);
    }
}
