<?php

namespace Tests\Feature;

use App\Models\Attendance;
use App\Models\Employee;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class AttendanceApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Employee $employee;

    protected function setUp(): void
    {
        parent::setUp();
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->employee = Employee::factory()->create(['tenant_id' => $this->tenant->id, 'user_id' => $this->user->id]);
    }

    public function test_can_check_in(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/attendance/check-in');

        $response->assertOk()
            ->assertJsonPath('data.employee_id', $this->employee->id)
            ->assertJsonStructure(['data' => ['check_in']]);
    }

    public function test_can_check_out(): void
    {
        Attendance::create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $this->employee->id,
            'date' => now()->toDateString(),
            'check_in' => '09:00:00',
        ]);

        $response = $this->actingAs($this->user)->postJson('/api/attendance/check-out');

        $response->assertOk()
            ->assertJsonStructure(['data' => ['check_out', 'hours_worked']]);
    }

    public function test_can_get_attendance_report(): void
    {
        Attendance::factory()->count(20)->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $this->employee->id,
        ]);

        $response = $this->actingAs($this->user)
            ->getJson('/api/attendance?month=' . now()->month . '&year=' . now()->year);

        $response->assertOk()
            ->assertJsonStructure(['data', 'stats']);
    }

    public function test_detects_late_check_in(): void
    {
        $this->travel(1)->hours(); // Set time to late

        $response = $this->actingAs($this->user)
            ->postJson('/api/attendance/check-in', ['check_in' => '10:30:00']);

        $response->assertOk();
        $this->assertEquals('late', Attendance::latest()->first()->status);
    }
}
