import { defineStore } from 'pinia';
import api from '../api';

export const useTasksStore = defineStore('tasks', {
    state: () => ({
        tasks: [],
        currentTask: null,
        isLoading: false,
        filters: {
            project_id: null,
            status_id: null,
            assignee_id: null,
            priority: null,
        },
    }),

    getters: {
        tasksByStatus: (state) => {
            return state.tasks.reduce((acc, task) => {
                const statusId = task.status_id;
                if (!acc[statusId]) acc[statusId] = [];
                acc[statusId].push(task);
                return acc;
            }, {});
        },

        overdueTasks: (state) => {
            const today = new Date().toISOString().split('T')[0];
            return state.tasks.filter(t => t.due_date && t.due_date < today && !t.completed_at);
        },

        completedTasks: (state) => state.tasks.filter(t => t.completed_at),
    },

    actions: {
        async fetchTasks(projectId) {
            this.isLoading = true;
            try {
                const params = { ...this.filters };
                if (projectId) params.project_id = projectId;
                const { data } = await api.get('/tasks', { params });
                this.tasks = data.data;
            } finally {
                this.isLoading = false;
            }
        },

        async fetchTask(id) {
            const { data } = await api.get(`/tasks/${id}`);
            this.currentTask = data.data;
            return this.currentTask;
        },

        async createTask(projectId, taskData) {
            const { data } = await api.post(`/projects/${projectId}/tasks`, taskData);
            this.tasks.push(data.data);
            return data.data;
        },

        async updateTask(id, taskData) {
            const { data } = await api.put(`/tasks/${id}`, taskData);
            const index = this.tasks.findIndex(t => t.id === id);
            if (index !== -1) this.tasks[index] = data.data;
            return data.data;
        },

        async moveTask(taskId, statusId) {
            await api.put(`/tasks/${taskId}/status`, { status_id: statusId });
            const task = this.tasks.find(t => t.id === taskId);
            if (task) task.status_id = statusId;
        },

        async completeTask(id) {
            const { data } = await api.post(`/tasks/${id}/complete`);
            const index = this.tasks.findIndex(t => t.id === id);
            if (index !== -1) this.tasks[index] = data.data;
        },

        async deleteTask(id) {
            await api.delete(`/tasks/${id}`);
            this.tasks = this.tasks.filter(t => t.id !== id);
        },
    },
});
