import { defineStore } from 'pinia';
import api from '../api';

export const useProjectStore = defineStore('projects', {
    state: () => ({
        projects: [],
        project: null,
        tasks: [],
        loading: false,
        error: null,
    }),

    getters: {
        activeProjects: (state) => state.projects.filter(p => p.status === 'active'),
        projectById: (state) => (id) => state.projects.find(p => p.id === id),
    },

    actions: {
        async fetchProjects(params = {}) {
            this.loading = true;
            try {
                const { data } = await api.get('/projects', { params });
                this.projects = data.data;
            } catch (error) {
                this.error = error.message;
            } finally {
                this.loading = false;
            }
        },

        async fetchProject(id) {
            this.loading = true;
            try {
                const { data } = await api.get(`/projects/${id}`);
                this.project = data.data;
                return this.project;
            } catch (error) {
                this.error = error.message;
            } finally {
                this.loading = false;
            }
        },

        async createProject(projectData) {
            const { data } = await api.post('/projects', projectData);
            this.projects.unshift(data.data);
            return data.data;
        },

        async updateProject(id, projectData) {
            const { data } = await api.put(`/projects/${id}`, projectData);
            const index = this.projects.findIndex(p => p.id === id);
            if (index !== -1) this.projects[index] = data.data;
            return data.data;
        },

        async fetchTasks(projectId, params = {}) {
            const { data } = await api.get(`/projects/${projectId}/tasks`, { params });
            this.tasks = data.data;
            return this.tasks;
        },

        async createTask(projectId, taskData) {
            const { data } = await api.post(`/projects/${projectId}/tasks`, taskData);
            this.tasks.push(data.data);
            return data.data;
        },

        async updateTask(taskId, taskData) {
            const { data } = await api.put(`/tasks/${taskId}`, taskData);
            const index = this.tasks.findIndex(t => t.id === taskId);
            if (index !== -1) this.tasks[index] = data.data;
            return data.data;
        },

        async moveTask(taskId, statusId) {
            const { data } = await api.put(`/tasks/${taskId}/status`, { status_id: statusId });
            const index = this.tasks.findIndex(t => t.id === taskId);
            if (index !== -1) this.tasks[index] = data.data;
            return data.data;
        },

        async completeTask(taskId) {
            const { data } = await api.post(`/tasks/${taskId}/complete`);
            const index = this.tasks.findIndex(t => t.id === taskId);
            if (index !== -1) this.tasks[index] = data.data;
            return data.data;
        },
    },
});
