# 🤖 BOT 7: COMMUNICATION MODULE

## 📋 YOUR RESPONSIBILITY
Build email, SMS, WhatsApp, phone calls, and unified inbox functionality.

---

## 📁 FILES TO CREATE

### 1. MODELS (12 files)
```
app/Models/
├── EmailAccount.php
├── EmailTemplate.php
├── Email.php
├── EmailAttachment.php
├── SmsMessage.php
├── WhatsappMessage.php
├── PhoneCall.php
├── Conversation.php
├── ConversationMessage.php
├── PhoneNumber.php
├── CallRecording.php
├── MessageTemplate.php
```

### 2. CONTROLLERS (8 files)
```
app/Http/Controllers/Api/
├── EmailAccountController.php
├── EmailController.php
├── EmailTemplateController.php
├── SmsController.php
├── WhatsappController.php
├── PhoneCallController.php
├── ConversationController.php (Unified Inbox)
├── WebhookController.php (Twilio/WhatsApp webhooks)
```

### 3. SERVICES (7 files)
```
app/Services/
├── EmailService.php
├── SmsService.php (Twilio)
├── WhatsappService.php (WhatsApp Business API)
├── TwilioService.php
├── PhoneCallService.php
├── ConversationService.php
├── MessageTemplateService.php
```

### 4. FORM REQUESTS (10 files)
```
app/Http/Requests/
├── Email/SendEmailRequest.php
├── Email/StoreEmailTemplateRequest.php
├── Sms/SendSmsRequest.php
├── Sms/BulkSmsRequest.php
├── Whatsapp/SendWhatsappRequest.php
├── Whatsapp/SendTemplateRequest.php
├── PhoneCall/InitiateCallRequest.php
├── Conversation/ReplyRequest.php
├── Conversation/AssignRequest.php
├── EmailAccount/StoreEmailAccountRequest.php
```

### 5. ROUTES (1 file)
```
routes/api/communication.php
```

### 6. JOBS (6 files)
```
app/Jobs/
├── SendEmail.php
├── SendBulkEmails.php
├── SendSms.php
├── SendWhatsappMessage.php
├── ProcessInboundEmail.php
├── SyncEmailInbox.php
```

### 7. EVENTS & LISTENERS (6 files)
```
app/Events/
├── EmailSent.php
├── EmailReceived.php
├── SmsSent.php
├── WhatsappMessageReceived.php
├── CallCompleted.php

app/Listeners/
├── LogCommunication.php
├── UpdateContactLastContacted.php
├── CreateActivityFromCommunication.php
```

### 8. NOTIFICATIONS (built into services)

### 9. TESTS (5 files)
```
tests/Feature/
├── EmailTest.php
├── SmsTest.php
├── WhatsappTest.php
├── PhoneCallTest.php
├── ConversationTest.php
```

---

## 📝 KEY CONTROLLERS

### EmailController.php
```php
public function index(Request $request);      // List emails
public function show(Email $email);           // Show email
public function compose();                    // Get compose data
public function send(SendEmailRequest $request);  // Send email
public function reply(Email $email, SendEmailRequest $request);
public function forward(Email $email, SendEmailRequest $request);
public function schedule(SendEmailRequest $request);  // Schedule email
public function cancelScheduled(Email $email);
public function markRead(Email $email);
public function markUnread(Email $email);
public function trash(Email $email);
public function threads(Request $request);    // Email threads
public function thread(string $threadId);     // Single thread
public function search(Request $request);     // Search emails
```

### SmsController.php
```php
public function index(Request $request);      // List SMS
public function send(SendSmsRequest $request);  // Send SMS
public function sendBulk(BulkSmsRequest $request);  // Bulk SMS
public function conversation(Contact $contact);  // SMS thread with contact
public function reply(Contact $contact, Request $request);
public function inbound(Request $request);    // Twilio webhook
```

### WhatsappController.php
```php
public function index(Request $request);      // List messages
public function send(SendWhatsappRequest $request);  // Send message
public function sendTemplate(SendTemplateRequest $request);  // Send template
public function sendMedia(Request $request);  // Send image/doc
public function conversation(Contact $contact);
public function reply(Contact $contact, Request $request);
public function templates();                  // List approved templates
public function webhook(Request $request);    // WhatsApp webhook
```

### ConversationController.php (Unified Inbox)
```php
public function index(Request $request);      // All conversations
public function show(Conversation $conversation);
public function reply(Conversation $conversation, ReplyRequest $request);
public function assign(Conversation $conversation, AssignRequest $request);
public function close(Conversation $conversation);
public function reopen(Conversation $conversation);
public function star(Conversation $conversation);
public function unread();                     // Unread count
public function filter(Request $request);     // Filter by channel/status
```

### PhoneCallController.php
```php
public function index(Request $request);      // Call log
public function initiate(InitiateCallRequest $request);  // Start call
public function incoming(Request $request);   // Twilio webhook
public function status(Request $request);     // Call status webhook
public function recording(PhoneCall $call);   // Get recording
public function transcription(PhoneCall $call);
public function logCall(Request $request);    // Manual call log
```

---

## 🔗 ROUTES

```php
// routes/api/communication.php

Route::middleware('auth:sanctum')->group(function () {
    // Email Accounts
    Route::apiResource('email-accounts', EmailAccountController::class);
    Route::post('email-accounts/{account}/test', [EmailAccountController::class, 'test']);
    Route::post('email-accounts/{account}/sync', [EmailAccountController::class, 'sync']);
    
    // Email Templates
    Route::apiResource('email-templates', EmailTemplateController::class);
    Route::post('email-templates/{template}/duplicate', [EmailTemplateController::class, 'duplicate']);
    
    // Emails
    Route::get('emails', [EmailController::class, 'index']);
    Route::get('emails/{email}', [EmailController::class, 'show']);
    Route::post('emails/send', [EmailController::class, 'send']);
    Route::post('emails/{email}/reply', [EmailController::class, 'reply']);
    Route::post('emails/{email}/forward', [EmailController::class, 'forward']);
    Route::post('emails/schedule', [EmailController::class, 'schedule']);
    Route::delete('emails/{email}/cancel', [EmailController::class, 'cancelScheduled']);
    Route::post('emails/{email}/read', [EmailController::class, 'markRead']);
    Route::post('emails/{email}/unread', [EmailController::class, 'markUnread']);
    Route::delete('emails/{email}', [EmailController::class, 'trash']);
    Route::get('email-threads', [EmailController::class, 'threads']);
    Route::get('email-threads/{threadId}', [EmailController::class, 'thread']);
    
    // SMS
    Route::get('sms', [SmsController::class, 'index']);
    Route::post('sms/send', [SmsController::class, 'send']);
    Route::post('sms/bulk', [SmsController::class, 'sendBulk']);
    Route::get('sms/conversation/{contact}', [SmsController::class, 'conversation']);
    Route::post('sms/conversation/{contact}/reply', [SmsController::class, 'reply']);
    
    // WhatsApp
    Route::get('whatsapp', [WhatsappController::class, 'index']);
    Route::post('whatsapp/send', [WhatsappController::class, 'send']);
    Route::post('whatsapp/template', [WhatsappController::class, 'sendTemplate']);
    Route::post('whatsapp/media', [WhatsappController::class, 'sendMedia']);
    Route::get('whatsapp/conversation/{contact}', [WhatsappController::class, 'conversation']);
    Route::post('whatsapp/conversation/{contact}/reply', [WhatsappController::class, 'reply']);
    Route::get('whatsapp/templates', [WhatsappController::class, 'templates']);
    
    // Phone Calls
    Route::get('calls', [PhoneCallController::class, 'index']);
    Route::post('calls/initiate', [PhoneCallController::class, 'initiate']);
    Route::get('calls/{call}/recording', [PhoneCallController::class, 'recording']);
    Route::get('calls/{call}/transcription', [PhoneCallController::class, 'transcription']);
    Route::post('calls/log', [PhoneCallController::class, 'logCall']);
    
    // Unified Inbox / Conversations
    Route::get('conversations', [ConversationController::class, 'index']);
    Route::get('conversations/{conversation}', [ConversationController::class, 'show']);
    Route::post('conversations/{conversation}/reply', [ConversationController::class, 'reply']);
    Route::post('conversations/{conversation}/assign', [ConversationController::class, 'assign']);
    Route::post('conversations/{conversation}/close', [ConversationController::class, 'close']);
    Route::post('conversations/{conversation}/reopen', [ConversationController::class, 'reopen']);
    Route::get('conversations-unread', [ConversationController::class, 'unread']);
});

// Webhooks (no auth)
Route::prefix('webhooks')->group(function () {
    Route::post('twilio/sms', [WebhookController::class, 'twilioSms']);
    Route::post('twilio/voice', [WebhookController::class, 'twilioVoice']);
    Route::post('twilio/status', [WebhookController::class, 'twilioStatus']);
    Route::post('whatsapp', [WebhookController::class, 'whatsapp']);
    Route::get('whatsapp', [WebhookController::class, 'whatsappVerify']);
    Route::post('sendgrid', [WebhookController::class, 'sendgrid']);
});
```

---

## 🔌 THIRD-PARTY INTEGRATIONS

### Twilio (SMS + Voice)
```php
// app/Services/TwilioService.php
class TwilioService
{
    public function sendSms(string $to, string $message): MessageInstance;
    public function sendBulkSms(array $recipients, string $message): array;
    public function initiateCall(string $to, string $from, string $url): CallInstance;
    public function getRecording(string $recordingSid): string;
    public function lookupNumber(string $number): array;
}
```

### WhatsApp Business API
```php
// app/Services/WhatsappService.php
class WhatsappService
{
    public function sendTextMessage(string $to, string $message): array;
    public function sendTemplate(string $to, string $templateName, array $params): array;
    public function sendMedia(string $to, string $mediaUrl, string $type): array;
    public function markAsRead(string $messageId): bool;
    public function getTemplates(): array;
}
```

---

## 🇧🇭 BAHRAIN-SPECIFIC

1. **WhatsApp**: Primary channel (wa.link/setupinbahrain)
2. **Phone Format**: +973 XXXX XXXX
3. **SMS**: English + Arabic support
4. **Email**: Arabic RTL support in templates

---

## ✅ COMPLETION CHECKLIST

- [ ] Email accounts + send/receive
- [ ] Email templates
- [ ] SMS via Twilio
- [ ] WhatsApp Business API
- [ ] Phone calls via Twilio
- [ ] Unified inbox (conversations)
- [ ] Webhooks for inbound
- [ ] Message threading
- [ ] All routes defined
- [ ] All tests passing

---

## 📤 OUTPUT

Save to: `/home/claude/business-platform/`
Update: `/home/claude/business-platform/docs/PROGRESS_BOT_7.md`
