# 🤖 BOT 4: ACCOUNTING - PAYABLES & LEDGER

## YOUR MISSION
Build payables & ledger: Bills, Vendors, Expenses, Chart of Accounts, Journal Entries, Bank Reconciliation.
**Est. Files:** 45 | **Est. Lines:** 10,000

---

## 📁 CREATE THESE FILES

### MIGRATIONS (8 files)
```
database/migrations/
├── 2024_01_04_000001_create_vendors_table.php
├── 2024_01_04_000002_create_bills_table.php
├── 2024_01_04_000003_create_bill_items_table.php
├── 2024_01_04_000004_create_bill_payments_table.php
├── 2024_01_04_000005_create_expenses_table.php
├── 2024_01_04_000006_create_chart_of_accounts_table.php
├── 2024_01_04_000007_create_journal_entries_table.php
├── 2024_01_04_000008_create_bank_accounts_table.php
├── 2024_01_04_000009_create_bank_transactions_table.php
```

### MODELS (12 files)
```
app/Models/
├── Vendor.php
├── Bill.php
├── BillItem.php
├── BillPayment.php
├── Expense.php
├── ExpenseCategory.php
├── ChartOfAccount.php
├── FiscalYear.php
├── JournalEntry.php
├── JournalEntryLine.php
├── BankAccount.php
├── BankTransaction.php
```

### CONTROLLERS (8 files)
```
app/Http/Controllers/Api/
├── VendorController.php
├── BillController.php
├── BillPaymentController.php
├── ExpenseController.php
├── ChartOfAccountController.php
├── JournalEntryController.php
├── BankAccountController.php
├── ReportController.php
```

### SERVICES (6 files)
```
app/Services/
├── VendorService.php
├── BillService.php
├── ExpenseService.php
├── JournalEntryService.php
├── BankingService.php
├── ReportService.php
```

### ROUTES (1 file)
```
routes/api/accounting.php
```

---

## 📝 KEY SPECIFICATIONS

### Migration: create_chart_of_accounts_table.php
```php
Schema::create('chart_of_accounts', function (Blueprint $table) {
    $table->uuid('id')->primary();
    $table->uuid('tenant_id');
    $table->uuid('parent_id')->nullable();
    
    $table->string('code', 20);
    $table->string('name');
    $table->text('description')->nullable();
    
    $table->enum('type', ['asset', 'liability', 'equity', 'revenue', 'expense']);
    $table->enum('detail_type', [
        // Assets
        'cash', 'bank', 'accounts_receivable', 'inventory', 'fixed_asset', 'other_asset',
        // Liabilities
        'accounts_payable', 'credit_card', 'tax_payable', 'loan', 'other_liability',
        // Equity
        'owner_equity', 'retained_earnings',
        // Revenue
        'sales', 'service_revenue', 'other_income',
        // Expense
        'cost_of_goods', 'operating_expense', 'payroll', 'other_expense'
    ]);
    
    $table->enum('normal_balance', ['debit', 'credit']);
    
    $table->decimal('opening_balance', 15, 3)->default(0);
    $table->decimal('current_balance', 15, 3)->default(0);
    
    $table->boolean('is_system')->default(false);
    $table->boolean('is_reconcilable')->default(false);
    $table->boolean('is_active')->default(true);
    
    $table->timestamps();
    $table->softDeletes();
    
    $table->foreign('tenant_id')->references('id')->on('tenants')->cascadeOnDelete();
    $table->unique(['tenant_id', 'code']);
});
```

### Migration: create_journal_entries_table.php
```php
Schema::create('journal_entries', function (Blueprint $table) {
    $table->uuid('id')->primary();
    $table->uuid('tenant_id');
    $table->uuid('fiscal_year_id');
    $table->uuid('created_by');
    
    $table->string('entry_number');
    $table->date('entry_date');
    $table->string('reference')->nullable();
    $table->text('description')->nullable();
    
    $table->nullableUuidMorphs('journalable'); // invoice, bill, expense, payment
    
    $table->decimal('total_debit', 15, 3)->default(0);
    $table->decimal('total_credit', 15, 3)->default(0);
    
    $table->enum('status', ['draft', 'posted', 'void'])->default('draft');
    $table->timestamp('posted_at')->nullable();
    $table->uuid('approved_by')->nullable();
    
    $table->timestamps();
    $table->softDeletes();
});

Schema::create('journal_entry_lines', function (Blueprint $table) {
    $table->uuid('id')->primary();
    $table->uuid('journal_entry_id');
    $table->uuid('account_id');
    $table->uuid('contact_id')->nullable();
    
    $table->text('description')->nullable();
    $table->decimal('debit', 15, 3)->default(0);
    $table->decimal('credit', 15, 3)->default(0);
    
    $table->timestamps();
    
    $table->foreign('journal_entry_id')->references('id')->on('journal_entries')->cascadeOnDelete();
    $table->foreign('account_id')->references('id')->on('chart_of_accounts');
});
```

### Model: ChartOfAccount.php
```php
<?php
namespace App\Models;

use App\Traits\BelongsToTenant;
use Illuminate\Database\Eloquent\{Model, SoftDeletes};
use Illuminate\Database\Eloquent\Concerns\HasUuids;

class ChartOfAccount extends Model
{
    use HasUuids, SoftDeletes, BelongsToTenant;
    
    protected $table = 'chart_of_accounts';
    protected $guarded = ['id'];
    
    protected $casts = [
        'opening_balance' => 'decimal:3',
        'current_balance' => 'decimal:3',
        'is_system' => 'boolean',
        'is_reconcilable' => 'boolean',
        'is_active' => 'boolean',
    ];
    
    // Relationships
    public function parent() { return $this->belongsTo(self::class, 'parent_id'); }
    public function children() { return $this->hasMany(self::class, 'parent_id'); }
    public function journalLines() { return $this->hasMany(JournalEntryLine::class, 'account_id'); }
    
    // Scopes
    public function scopeOfType($q, string $type) { return $q->where('type', $type); }
    public function scopeActive($q) { return $q->where('is_active', true); }
    
    // Methods
    public function isDebitNormal(): bool { return $this->normal_balance === 'debit'; }
    public function isCreditNormal(): bool { return $this->normal_balance === 'credit'; }
    
    public function updateBalance(float $debit, float $credit): void
    {
        if ($this->isDebitNormal()) {
            $this->current_balance += ($debit - $credit);
        } else {
            $this->current_balance += ($credit - $debit);
        }
        $this->save();
    }
    
    public static function getByDetailType(string $detailType): ?self
    {
        return static::where('detail_type', $detailType)->first();
    }
}
```

### Model: JournalEntry.php
```php
<?php
namespace App\Models;

use App\Traits\BelongsToTenant;
use Illuminate\Database\Eloquent\{Model, SoftDeletes};
use Illuminate\Database\Eloquent\Concerns\HasUuids;
use Illuminate\Support\Facades\DB;

class JournalEntry extends Model
{
    use HasUuids, SoftDeletes, BelongsToTenant;
    
    protected $guarded = ['id'];
    
    protected $casts = [
        'entry_date' => 'date',
        'posted_at' => 'datetime',
        'total_debit' => 'decimal:3',
        'total_credit' => 'decimal:3',
    ];
    
    // Relationships
    public function fiscalYear() { return $this->belongsTo(FiscalYear::class); }
    public function lines() { return $this->hasMany(JournalEntryLine::class); }
    public function journalable() { return $this->morphTo(); }
    public function createdBy() { return $this->belongsTo(User::class, 'created_by'); }
    
    // Methods
    public function isBalanced(): bool
    {
        return bccomp($this->total_debit, $this->total_credit, 3) === 0;
    }
    
    public function calculateTotals(): void
    {
        $this->total_debit = $this->lines()->sum('debit');
        $this->total_credit = $this->lines()->sum('credit');
        $this->save();
    }
    
    public function post(): void
    {
        if (!$this->isBalanced()) {
            throw new \Exception('Journal entry is not balanced');
        }
        
        DB::transaction(function () {
            foreach ($this->lines as $line) {
                $line->account->updateBalance($line->debit, $line->credit);
            }
            
            $this->update([
                'status' => 'posted',
                'posted_at' => now(),
                'approved_by' => auth()->id(),
            ]);
        });
    }
    
    public function void(): void
    {
        DB::transaction(function () {
            // Reverse account balances
            foreach ($this->lines as $line) {
                $line->account->updateBalance(-$line->debit, -$line->credit);
            }
            
            $this->update(['status' => 'void']);
        });
    }
}
```

### Service: JournalEntryService.php
```php
<?php
namespace App\Services;

use App\Models\{JournalEntry, JournalEntryLine, ChartOfAccount, Invoice, Bill, Expense, Payment};
use Illuminate\Support\Facades\DB;

class JournalEntryService
{
    public function createFromInvoice(Invoice $invoice): JournalEntry
    {
        return DB::transaction(function () use ($invoice) {
            $entry = JournalEntry::create([
                'tenant_id' => $invoice->tenant_id,
                'fiscal_year_id' => $this->getCurrentFiscalYear($invoice->tenant_id)->id,
                'created_by' => auth()->id(),
                'entry_date' => $invoice->invoice_date,
                'reference' => $invoice->invoice_number,
                'description' => "Invoice {$invoice->invoice_number}",
                'journalable_type' => Invoice::class,
                'journalable_id' => $invoice->id,
            ]);
            
            // Debit: Accounts Receivable
            $entry->lines()->create([
                'account_id' => ChartOfAccount::getByDetailType('accounts_receivable')->id,
                'contact_id' => $invoice->contact_id,
                'description' => "AR - {$invoice->contact->display_name}",
                'debit' => $invoice->total,
                'credit' => 0,
            ]);
            
            // Credit: Sales Revenue
            $entry->lines()->create([
                'account_id' => ChartOfAccount::getByDetailType('sales')->id,
                'description' => 'Sales Revenue',
                'debit' => 0,
                'credit' => $invoice->subtotal,
            ]);
            
            // Credit: Tax Payable (if applicable)
            if ($invoice->tax_amount > 0) {
                $entry->lines()->create([
                    'account_id' => ChartOfAccount::getByDetailType('tax_payable')->id,
                    'description' => 'VAT Payable',
                    'debit' => 0,
                    'credit' => $invoice->tax_amount,
                ]);
            }
            
            $entry->calculateTotals();
            $entry->post();
            
            return $entry;
        });
    }
    
    public function createFromPaymentReceived(Payment $payment): JournalEntry
    {
        return DB::transaction(function () use ($payment) {
            $entry = JournalEntry::create([
                'tenant_id' => $payment->tenant_id,
                'fiscal_year_id' => $this->getCurrentFiscalYear($payment->tenant_id)->id,
                'created_by' => auth()->id(),
                'entry_date' => $payment->payment_date,
                'reference' => $payment->reference,
                'description' => "Payment received - {$payment->invoice->invoice_number}",
                'journalable_type' => Payment::class,
                'journalable_id' => $payment->id,
            ]);
            
            // Debit: Bank/Cash
            $entry->lines()->create([
                'account_id' => $payment->bank_account_id 
                    ? $payment->bankAccount->account_id 
                    : ChartOfAccount::getByDetailType('cash')->id,
                'description' => 'Payment received',
                'debit' => $payment->amount,
                'credit' => 0,
            ]);
            
            // Credit: Accounts Receivable
            $entry->lines()->create([
                'account_id' => ChartOfAccount::getByDetailType('accounts_receivable')->id,
                'contact_id' => $payment->contact_id,
                'description' => "AR - {$payment->contact->display_name}",
                'debit' => 0,
                'credit' => $payment->amount,
            ]);
            
            $entry->calculateTotals();
            $entry->post();
            
            return $entry;
        });
    }
    
    public function createFromBill(Bill $bill): JournalEntry
    {
        // Similar to invoice but reverse (AP and Expense)
    }
    
    public function createFromExpense(Expense $expense): JournalEntry
    {
        // Debit Expense account, Credit Cash/AP
    }
    
    private function getCurrentFiscalYear(string $tenantId)
    {
        return \App\Models\FiscalYear::where('tenant_id', $tenantId)
            ->where('is_closed', false)
            ->first();
    }
}
```

### Routes: routes/api/accounting.php
```php
<?php
use Illuminate\Support\Facades\Route;

Route::middleware('auth:sanctum')->group(function () {
    // Vendors
    Route::apiResource('vendors', VendorController::class);
    Route::get('vendors/{vendor}/bills', [VendorController::class, 'bills']);
    Route::get('vendors/{vendor}/statement', [VendorController::class, 'statement']);
    
    // Bills
    Route::apiResource('bills', BillController::class);
    Route::post('bills/{bill}/payment', [BillController::class, 'recordPayment']);
    Route::post('bills/{bill}/void', [BillController::class, 'void']);
    Route::get('bills-summary', [BillController::class, 'summary']);
    Route::get('bills-aging', [BillController::class, 'aging']);
    
    // Expenses
    Route::apiResource('expenses', ExpenseController::class);
    Route::post('expenses/{expense}/approve', [ExpenseController::class, 'approve']);
    Route::post('expenses/{expense}/reject', [ExpenseController::class, 'reject']);
    
    // Chart of Accounts
    Route::apiResource('accounts', ChartOfAccountController::class);
    Route::get('accounts/{account}/ledger', [ChartOfAccountController::class, 'ledger']);
    
    // Journal Entries
    Route::apiResource('journal-entries', JournalEntryController::class);
    Route::post('journal-entries/{entry}/post', [JournalEntryController::class, 'post']);
    Route::post('journal-entries/{entry}/void', [JournalEntryController::class, 'void']);
    
    // Bank Accounts
    Route::apiResource('bank-accounts', BankAccountController::class);
    Route::get('bank-accounts/{account}/transactions', [BankAccountController::class, 'transactions']);
    Route::post('bank-accounts/{account}/reconcile', [BankAccountController::class, 'reconcile']);
    
    // Reports
    Route::prefix('reports')->group(function () {
        Route::get('trial-balance', [ReportController::class, 'trialBalance']);
        Route::get('balance-sheet', [ReportController::class, 'balanceSheet']);
        Route::get('income-statement', [ReportController::class, 'incomeStatement']);
        Route::get('cash-flow', [ReportController::class, 'cashFlow']);
        Route::get('general-ledger', [ReportController::class, 'generalLedger']);
    });
});
```

---

## ✅ COMPLETION CHECKLIST
- [ ] All migrations
- [ ] All models
- [ ] Double-entry accounting logic
- [ ] Auto journal entries from invoices/bills/payments
- [ ] Chart of accounts with hierarchy
- [ ] Financial reports (P&L, Balance Sheet)
- [ ] Bank reconciliation
- [ ] All routes
- [ ] Tests

---

## 🔗 DEPENDENCIES
- Bot 1: Tenant, User
- Bot 2: Contact
- Bot 3: Invoice, Payment

## 📤 OUTPUT
Save to: `/home/claude/business-platform/`
