# 🤖 BOT 3: ACCOUNTING - RECEIVABLES

## YOUR MISSION
Build invoicing: Invoices, Estimates, Payments Received, Recurring Invoices.
**Est. Files:** 45 | **Est. Lines:** 10,000

---

## 📁 CREATE THESE FILES

### MIGRATIONS (6 files)
```
database/migrations/
├── 2024_01_03_000001_create_currencies_table.php
├── 2024_01_03_000002_create_tax_rates_table.php
├── 2024_01_03_000003_create_invoices_table.php
├── 2024_01_03_000004_create_invoice_items_table.php
├── 2024_01_03_000005_create_estimates_table.php
├── 2024_01_03_000006_create_payments_table.php
├── 2024_01_03_000007_create_recurring_invoices_table.php
```

### MODELS (10 files)
```
app/Models/
├── Currency.php
├── TaxRate.php
├── Invoice.php
├── InvoiceItem.php
├── Estimate.php
├── EstimateItem.php
├── Payment.php
├── RecurringInvoice.php
├── PaymentTerm.php
├── InvoiceTemplate.php
```

### CONTROLLERS (6 files)
```
app/Http/Controllers/Api/
├── InvoiceController.php
├── EstimateController.php
├── PaymentController.php
├── RecurringInvoiceController.php
├── TaxRateController.php
├── CurrencyController.php
```

### SERVICES (5 files)
```
app/Services/
├── InvoiceService.php
├── EstimateService.php
├── PaymentService.php
├── RecurringInvoiceService.php
├── InvoicePdfService.php
```

### REQUESTS (10 files)
```
app/Http/Requests/Invoice/
├── StoreInvoiceRequest.php
├── UpdateInvoiceRequest.php
├── SendInvoiceRequest.php

app/Http/Requests/Estimate/
├── StoreEstimateRequest.php
├── UpdateEstimateRequest.php

app/Http/Requests/Payment/
├── StorePaymentRequest.php
├── RefundPaymentRequest.php

app/Http/Requests/RecurringInvoice/
├── StoreRecurringInvoiceRequest.php

app/Http/Requests/TaxRate/
├── StoreTaxRateRequest.php
```

### ROUTES (1 file)
```
routes/api/invoicing.php
```

### JOBS (4 files)
```
app/Jobs/
├── SendInvoiceEmail.php
├── GenerateRecurringInvoices.php
├── SendPaymentReminder.php
├── MarkOverdueInvoices.php
```

### VIEWS (2 files)
```
resources/views/invoices/
├── pdf.blade.php
├── email.blade.php
```

### TESTS (5 files)
```
tests/Feature/
├── InvoiceTest.php
├── EstimateTest.php
├── PaymentTest.php
├── RecurringInvoiceTest.php
├── InvoicePdfTest.php
```

---

## 📝 KEY SPECIFICATIONS

### Migration: create_invoices_table.php
```php
Schema::create('invoices', function (Blueprint $table) {
    $table->uuid('id')->primary();
    $table->uuid('tenant_id');
    $table->uuid('contact_id');
    $table->uuid('created_by');
    
    $table->string('invoice_number')->unique();
    $table->string('order_number')->nullable();
    $table->string('po_number')->nullable();
    
    $table->enum('status', ['draft', 'sent', 'viewed', 'partial', 'paid', 'overdue', 'cancelled', 'refunded'])->default('draft');
    
    $table->date('invoice_date');
    $table->date('due_date');
    
    $table->string('currency', 3)->default('BHD');
    $table->decimal('exchange_rate', 15, 6)->default(1);
    
    $table->decimal('subtotal', 15, 3)->default(0);
    $table->decimal('discount_amount', 15, 3)->default(0);
    $table->string('discount_type')->nullable(); // fixed, percentage
    $table->decimal('tax_amount', 15, 3)->default(0);
    $table->decimal('total', 15, 3)->default(0);
    $table->decimal('amount_paid', 15, 3)->default(0);
    $table->decimal('amount_due', 15, 3)->default(0);
    
    $table->json('billing_address')->nullable();
    $table->json('shipping_address')->nullable();
    
    $table->text('notes')->nullable();
    $table->text('terms')->nullable();
    $table->string('footer')->nullable();
    
    $table->timestamp('sent_at')->nullable();
    $table->timestamp('viewed_at')->nullable();
    $table->timestamp('paid_at')->nullable();
    
    $table->timestamps();
    $table->softDeletes();
    
    $table->foreign('tenant_id')->references('id')->on('tenants')->cascadeOnDelete();
    $table->foreign('contact_id')->references('id')->on('contacts');
});
```

### Model: Invoice.php
```php
<?php
namespace App\Models;

use App\Traits\BelongsToTenant;
use Illuminate\Database\Eloquent\{Model, SoftDeletes};
use Illuminate\Database\Eloquent\Concerns\HasUuids;

class Invoice extends Model
{
    use HasUuids, SoftDeletes, BelongsToTenant;
    
    protected $guarded = ['id'];
    
    protected $casts = [
        'invoice_date' => 'date',
        'due_date' => 'date',
        'sent_at' => 'datetime',
        'viewed_at' => 'datetime',
        'paid_at' => 'datetime',
        'billing_address' => 'array',
        'shipping_address' => 'array',
        'subtotal' => 'decimal:3',
        'tax_amount' => 'decimal:3',
        'total' => 'decimal:3',
        'amount_paid' => 'decimal:3',
        'amount_due' => 'decimal:3',
    ];
    
    protected static function booted(): void
    {
        static::creating(function ($invoice) {
            if (!$invoice->invoice_number) {
                $invoice->invoice_number = static::generateNumber($invoice->tenant_id);
            }
        });
    }
    
    // Relationships
    public function contact() { return $this->belongsTo(Contact::class); }
    public function items() { return $this->hasMany(InvoiceItem::class); }
    public function payments() { return $this->hasMany(Payment::class); }
    public function createdBy() { return $this->belongsTo(User::class, 'created_by'); }
    
    // Scopes
    public function scopeDraft($q) { return $q->where('status', 'draft'); }
    public function scopeSent($q) { return $q->whereIn('status', ['sent', 'viewed', 'partial']); }
    public function scopePaid($q) { return $q->where('status', 'paid'); }
    public function scopeOverdue($q) { return $q->where('status', 'overdue'); }
    public function scopeUnpaid($q) { return $q->whereIn('status', ['sent', 'viewed', 'partial', 'overdue']); }
    
    // Methods
    public static function generateNumber(string $tenantId): string
    {
        $last = static::where('tenant_id', $tenantId)
            ->orderByDesc('invoice_number')
            ->value('invoice_number');
        
        $num = $last ? ((int) substr($last, 4)) + 1 : 1;
        return 'INV-' . str_pad($num, 5, '0', STR_PAD_LEFT);
    }
    
    public function calculateTotals(): void
    {
        $subtotal = $this->items->sum(fn($item) => $item->quantity * $item->unit_price);
        $taxAmount = $this->items->sum('tax_amount');
        $total = $subtotal + $taxAmount - $this->discount_amount;
        $amountDue = $total - $this->amount_paid;
        
        $this->update([
            'subtotal' => $subtotal,
            'tax_amount' => $taxAmount,
            'total' => $total,
            'amount_due' => $amountDue,
        ]);
    }
    
    public function recordPayment(float $amount, array $data = []): Payment
    {
        $payment = $this->payments()->create([
            'tenant_id' => $this->tenant_id,
            'contact_id' => $this->contact_id,
            'amount' => $amount,
            'currency' => $this->currency,
            'payment_date' => $data['payment_date'] ?? now(),
            'payment_method' => $data['payment_method'] ?? 'bank_transfer',
            'reference' => $data['reference'] ?? null,
            'notes' => $data['notes'] ?? null,
        ]);
        
        $this->amount_paid += $amount;
        $this->amount_due = $this->total - $this->amount_paid;
        
        if ($this->amount_due <= 0) {
            $this->status = 'paid';
            $this->paid_at = now();
        } else {
            $this->status = 'partial';
        }
        
        $this->save();
        
        return $payment;
    }
    
    public function isPaid(): bool { return $this->status === 'paid'; }
    public function isOverdue(): bool { return $this->due_date->isPast() && !$this->isPaid(); }
}
```

### Service: InvoiceService.php
```php
<?php
namespace App\Services;

use App\Models\{Invoice, InvoiceItem};
use Illuminate\Support\Facades\DB;

class InvoiceService
{
    public function create(array $data): Invoice
    {
        return DB::transaction(function () use ($data) {
            $invoice = Invoice::create([
                'contact_id' => $data['contact_id'],
                'invoice_date' => $data['invoice_date'],
                'due_date' => $data['due_date'],
                'currency' => $data['currency'] ?? 'BHD',
                'notes' => $data['notes'] ?? null,
                'terms' => $data['terms'] ?? null,
                'billing_address' => $data['billing_address'] ?? null,
                'created_by' => auth()->id(),
            ]);
            
            foreach ($data['items'] as $item) {
                $invoice->items()->create([
                    'product_id' => $item['product_id'] ?? null,
                    'name' => $item['name'],
                    'description' => $item['description'] ?? null,
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'tax_rate_id' => $item['tax_rate_id'] ?? null,
                    'tax_amount' => $this->calculateTax($item),
                ]);
            }
            
            $invoice->calculateTotals();
            
            return $invoice->fresh(['items', 'contact']);
        });
    }
    
    public function update(Invoice $invoice, array $data): Invoice
    {
        return DB::transaction(function () use ($invoice, $data) {
            $invoice->update([
                'contact_id' => $data['contact_id'] ?? $invoice->contact_id,
                'invoice_date' => $data['invoice_date'] ?? $invoice->invoice_date,
                'due_date' => $data['due_date'] ?? $invoice->due_date,
                'notes' => $data['notes'] ?? $invoice->notes,
                'terms' => $data['terms'] ?? $invoice->terms,
            ]);
            
            if (isset($data['items'])) {
                $invoice->items()->delete();
                foreach ($data['items'] as $item) {
                    $invoice->items()->create([
                        'product_id' => $item['product_id'] ?? null,
                        'name' => $item['name'],
                        'description' => $item['description'] ?? null,
                        'quantity' => $item['quantity'],
                        'unit_price' => $item['unit_price'],
                        'tax_rate_id' => $item['tax_rate_id'] ?? null,
                        'tax_amount' => $this->calculateTax($item),
                    ]);
                }
                $invoice->calculateTotals();
            }
            
            return $invoice->fresh(['items', 'contact']);
        });
    }
    
    public function send(Invoice $invoice, array $recipients = []): void
    {
        // Update status
        $invoice->update([
            'status' => 'sent',
            'sent_at' => now(),
        ]);
        
        // Dispatch email job
        dispatch(new \App\Jobs\SendInvoiceEmail($invoice, $recipients));
    }
    
    public function void(Invoice $invoice): void
    {
        $invoice->update(['status' => 'cancelled']);
        
        // Create reversing journal entry if needed
    }
    
    public function duplicate(Invoice $invoice): Invoice
    {
        $new = $invoice->replicate(['invoice_number', 'status', 'sent_at', 'paid_at']);
        $new->status = 'draft';
        $new->invoice_date = now();
        $new->due_date = now()->addDays(30);
        $new->amount_paid = 0;
        $new->save();
        
        foreach ($invoice->items as $item) {
            $new->items()->create($item->toArray());
        }
        
        $new->calculateTotals();
        
        return $new;
    }
    
    private function calculateTax(array $item): float
    {
        if (!isset($item['tax_rate_id'])) return 0;
        
        $taxRate = \App\Models\TaxRate::find($item['tax_rate_id']);
        if (!$taxRate) return 0;
        
        $lineTotal = $item['quantity'] * $item['unit_price'];
        return $lineTotal * ($taxRate->rate / 100);
    }
}
```

### Routes: routes/api/invoicing.php
```php
<?php
use Illuminate\Support\Facades\Route;

Route::middleware('auth:sanctum')->group(function () {
    // Invoices
    Route::apiResource('invoices', InvoiceController::class);
    Route::post('invoices/{invoice}/send', [InvoiceController::class, 'send']);
    Route::get('invoices/{invoice}/pdf', [InvoiceController::class, 'pdf']);
    Route::post('invoices/{invoice}/payment', [InvoiceController::class, 'recordPayment']);
    Route::post('invoices/{invoice}/void', [InvoiceController::class, 'void']);
    Route::post('invoices/{invoice}/duplicate', [InvoiceController::class, 'duplicate']);
    Route::get('invoices-summary', [InvoiceController::class, 'summary']);
    Route::get('invoices-aging', [InvoiceController::class, 'aging']);
    
    // Estimates
    Route::apiResource('estimates', EstimateController::class);
    Route::post('estimates/{estimate}/send', [EstimateController::class, 'send']);
    Route::get('estimates/{estimate}/pdf', [EstimateController::class, 'pdf']);
    Route::post('estimates/{estimate}/accept', [EstimateController::class, 'accept']);
    Route::post('estimates/{estimate}/convert', [EstimateController::class, 'convertToInvoice']);
    
    // Payments
    Route::apiResource('payments', PaymentController::class);
    Route::post('payments/{payment}/refund', [PaymentController::class, 'refund']);
    
    // Recurring Invoices
    Route::apiResource('recurring-invoices', RecurringInvoiceController::class);
    Route::post('recurring-invoices/{recurring}/pause', [RecurringInvoiceController::class, 'pause']);
    Route::post('recurring-invoices/{recurring}/generate', [RecurringInvoiceController::class, 'generateNow']);
    
    // Tax Rates
    Route::apiResource('tax-rates', TaxRateController::class);
    
    // Currencies
    Route::apiResource('currencies', CurrencyController::class);
});

// Public invoice view
Route::get('invoices/{invoice}/view/{token}', [PublicInvoiceController::class, 'show']);
```

---

## 🇧🇭 BAHRAIN SPECIFICS
- Currency: BHD (3 decimal places)
- VAT: 10%
- CR Number on invoices
- Arabic support in PDF

---

## ✅ COMPLETION CHECKLIST
- [ ] All migrations
- [ ] All models with relationships
- [ ] All controllers
- [ ] InvoiceService with full logic
- [ ] PDF generation
- [ ] Email sending
- [ ] Payment recording
- [ ] Recurring invoice scheduler
- [ ] Routes complete
- [ ] Tests passing

---

## 🔗 DEPENDENCIES
- Bot 1: Tenant, User
- Bot 2: Contact model

## 📤 OUTPUT
Save to: `/home/claude/business-platform/`
Update: `/home/claude/business-platform/docs/PROGRESS_BOT_3.md`
