# 🤖 BOT 1: CORE & AUTHENTICATION

## YOUR MISSION
Build the foundation: multi-tenancy, auth, users, teams, permissions.
**Est. Files:** 45 | **Est. Lines:** 8,000

---

## 📁 CREATE THESE FILES

### MIGRATIONS (6 files)
```
database/migrations/
├── 2024_01_01_000001_create_tenants_table.php
├── 2024_01_01_000002_create_users_table.php
├── 2024_01_01_000003_create_teams_table.php
├── 2024_01_01_000004_create_team_members_table.php
├── 2024_01_01_000005_create_invitations_table.php
├── 2024_01_01_000006_create_user_settings_table.php
```

### MODELS (8 files)
```
app/Models/
├── Tenant.php
├── User.php
├── Team.php
├── TeamMember.php
├── Invitation.php
├── UserSetting.php
├── Permission.php
├── Role.php
```

### CONTROLLERS (7 files)
```
app/Http/Controllers/Api/
├── AuthController.php
├── ProfileController.php
├── TenantController.php
├── UserController.php
├── TeamController.php
├── RoleController.php
├── InvitationController.php
```

### SERVICES (5 files)
```
app/Services/
├── AuthService.php
├── TenantService.php
├── UserService.php
├── TeamService.php
├── InvitationService.php
```

### REQUESTS (12 files)
```
app/Http/Requests/Auth/
├── LoginRequest.php
├── RegisterRequest.php
├── ForgotPasswordRequest.php
├── ResetPasswordRequest.php

app/Http/Requests/User/
├── StoreUserRequest.php
├── UpdateUserRequest.php

app/Http/Requests/Team/
├── StoreTeamRequest.php
├── UpdateTeamRequest.php

app/Http/Requests/Tenant/
├── UpdateTenantRequest.php

app/Http/Requests/Invitation/
├── StoreInvitationRequest.php
├── AcceptInvitationRequest.php
```

### MIDDLEWARE (3 files)
```
app/Http/Middleware/
├── EnsureTenantAccess.php
├── CheckPermission.php
├── TrackLastActivity.php
```

### TRAITS (2 files)
```
app/Traits/
├── BelongsToTenant.php
├── HasPermissions.php
```

### ROUTES (1 file)
```
routes/api/auth.php
```

### SEEDERS (2 files)
```
database/seeders/
├── RolePermissionSeeder.php
├── TenantSeeder.php
```

### TESTS (5 files)
```
tests/Feature/
├── AuthTest.php
├── TenantTest.php
├── UserTest.php
├── TeamTest.php
├── PermissionTest.php
```

---

## 📝 DETAILED SPECIFICATIONS

### Migration: create_tenants_table.php
```php
Schema::create('tenants', function (Blueprint $table) {
    $table->uuid('id')->primary();
    $table->string('name');
    $table->string('slug')->unique();
    $table->string('email');
    $table->string('phone')->nullable();
    $table->text('address')->nullable();
    $table->string('city')->nullable();
    $table->string('country')->default('BH');
    $table->string('currency', 3)->default('BHD');
    $table->string('timezone')->default('Asia/Bahrain');
    $table->string('logo_path')->nullable();
    $table->string('tax_number')->nullable(); // CR Number
    $table->string('vat_number')->nullable();
    $table->json('settings')->nullable();
    $table->string('plan')->default('free');
    $table->timestamp('trial_ends_at')->nullable();
    $table->timestamps();
    $table->softDeletes();
});
```

### Migration: create_users_table.php
```php
Schema::create('users', function (Blueprint $table) {
    $table->uuid('id')->primary();
    $table->uuid('tenant_id');
    $table->string('name');
    $table->string('email');
    $table->string('phone')->nullable();
    $table->timestamp('email_verified_at')->nullable();
    $table->string('password');
    $table->string('avatar_path')->nullable();
    $table->string('role')->default('member'); // owner, admin, member
    $table->boolean('is_active')->default(true);
    $table->timestamp('last_login_at')->nullable();
    $table->string('last_login_ip')->nullable();
    $table->json('preferences')->nullable();
    $table->rememberToken();
    $table->timestamps();
    $table->softDeletes();
    
    $table->foreign('tenant_id')->references('id')->on('tenants')->cascadeOnDelete();
    $table->unique(['tenant_id', 'email']);
});
```

### Model: Tenant.php
```php
<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Concerns\HasUuids;

class Tenant extends Model
{
    use HasUuids, SoftDeletes;
    
    protected $guarded = ['id'];
    
    protected $casts = [
        'settings' => 'array',
        'trial_ends_at' => 'datetime',
    ];
    
    public function users() { return $this->hasMany(User::class); }
    public function teams() { return $this->hasMany(Team::class); }
    public function owner() { return $this->hasOne(User::class)->where('role', 'owner'); }
    
    public function isOnTrial(): bool {
        return $this->trial_ends_at && $this->trial_ends_at->isFuture();
    }
    
    public function getSetting(string $key, $default = null) {
        return data_get($this->settings, $key, $default);
    }
}
```

### Model: User.php
```php
<?php
namespace App\Models;

use Illuminate\Foundation\Auth\User as Authenticatable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Concerns\HasUuids;
use App\Traits\BelongsToTenant;

class User extends Authenticatable
{
    use HasApiTokens, HasUuids, SoftDeletes, BelongsToTenant;
    
    protected $guarded = ['id'];
    protected $hidden = ['password', 'remember_token'];
    
    protected $casts = [
        'email_verified_at' => 'datetime',
        'last_login_at' => 'datetime',
        'is_active' => 'boolean',
        'preferences' => 'array',
    ];
    
    public function tenant() { return $this->belongsTo(Tenant::class); }
    public function teams() { return $this->belongsToMany(Team::class, 'team_members'); }
    public function ownedTeams() { return $this->hasMany(Team::class, 'owner_id'); }
    
    public function isOwner(): bool { return $this->role === 'owner'; }
    public function isAdmin(): bool { return in_array($this->role, ['owner', 'admin']); }
    
    public function hasPermission(string $permission): bool {
        if ($this->isOwner()) return true;
        // Check role permissions
        return false; // Implement based on your permission system
    }
}
```

### Trait: BelongsToTenant.php
```php
<?php
namespace App\Traits;

use App\Models\Tenant;
use Illuminate\Database\Eloquent\Builder;

trait BelongsToTenant
{
    protected static function bootBelongsToTenant(): void
    {
        // Auto-scope queries to current tenant
        static::addGlobalScope('tenant', function (Builder $builder) {
            if ($tenant = app('currentTenant')) {
                $builder->where('tenant_id', $tenant->id);
            }
        });
        
        // Auto-set tenant_id on create
        static::creating(function ($model) {
            if (!$model->tenant_id && $tenant = app('currentTenant')) {
                $model->tenant_id = $tenant->id;
            }
        });
    }
    
    public function tenant() {
        return $this->belongsTo(Tenant::class);
    }
}
```

### Controller: AuthController.php
```php
<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\{LoginRequest, RegisterRequest, ForgotPasswordRequest, ResetPasswordRequest};
use App\Services\AuthService;
use Illuminate\Http\JsonResponse;

class AuthController extends Controller
{
    public function __construct(private AuthService $authService) {}
    
    public function register(RegisterRequest $request): JsonResponse
    {
        $result = $this->authService->register($request->validated());
        return response()->json([
            'success' => true,
            'message' => 'Registration successful',
            'data' => $result
        ], 201);
    }
    
    public function login(LoginRequest $request): JsonResponse
    {
        $result = $this->authService->login($request->validated());
        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'data' => $result
        ]);
    }
    
    public function logout(): JsonResponse
    {
        auth()->user()->currentAccessToken()->delete();
        return response()->json([
            'success' => true,
            'message' => 'Logged out successfully'
        ]);
    }
    
    public function me(): JsonResponse
    {
        return response()->json([
            'success' => true,
            'data' => [
                'user' => auth()->user()->load('tenant'),
            ]
        ]);
    }
    
    public function forgotPassword(ForgotPasswordRequest $request): JsonResponse
    {
        $this->authService->sendPasswordResetLink($request->email);
        return response()->json([
            'success' => true,
            'message' => 'Password reset link sent'
        ]);
    }
    
    public function resetPassword(ResetPasswordRequest $request): JsonResponse
    {
        $this->authService->resetPassword($request->validated());
        return response()->json([
            'success' => true,
            'message' => 'Password reset successful'
        ]);
    }
}
```

### Service: AuthService.php
```php
<?php
namespace App\Services;

use App\Models\{Tenant, User};
use Illuminate\Support\Facades\{Hash, Password, DB};
use Illuminate\Validation\ValidationException;

class AuthService
{
    public function register(array $data): array
    {
        return DB::transaction(function () use ($data) {
            // Create tenant
            $tenant = Tenant::create([
                'name' => $data['company_name'],
                'slug' => \Str::slug($data['company_name']),
                'email' => $data['email'],
            ]);
            
            // Create owner user
            $user = User::create([
                'tenant_id' => $tenant->id,
                'name' => $data['name'],
                'email' => $data['email'],
                'password' => Hash::make($data['password']),
                'role' => 'owner',
            ]);
            
            // Create token
            $token = $user->createToken('auth-token')->plainTextToken;
            
            return [
                'user' => $user,
                'tenant' => $tenant,
                'token' => $token,
            ];
        });
    }
    
    public function login(array $credentials): array
    {
        $user = User::where('email', $credentials['email'])->first();
        
        if (!$user || !Hash::check($credentials['password'], $user->password)) {
            throw ValidationException::withMessages([
                'email' => ['Invalid credentials'],
            ]);
        }
        
        if (!$user->is_active) {
            throw ValidationException::withMessages([
                'email' => ['Account is deactivated'],
            ]);
        }
        
        $user->update([
            'last_login_at' => now(),
            'last_login_ip' => request()->ip(),
        ]);
        
        $token = $user->createToken('auth-token')->plainTextToken;
        
        return [
            'user' => $user->load('tenant'),
            'token' => $token,
        ];
    }
    
    public function sendPasswordResetLink(string $email): void
    {
        Password::sendResetLink(['email' => $email]);
    }
    
    public function resetPassword(array $data): void
    {
        Password::reset($data, function ($user, $password) {
            $user->update(['password' => Hash::make($password)]);
        });
    }
}
```

### Routes: routes/api/auth.php
```php
<?php
use App\Http\Controllers\Api\{
    AuthController,
    ProfileController,
    TenantController,
    UserController,
    TeamController,
    RoleController,
    InvitationController
};
use Illuminate\Support\Facades\Route;

// Public routes
Route::post('register', [AuthController::class, 'register']);
Route::post('login', [AuthController::class, 'login']);
Route::post('forgot-password', [AuthController::class, 'forgotPassword']);
Route::post('reset-password', [AuthController::class, 'resetPassword']);
Route::get('invitations/{token}', [InvitationController::class, 'show']);
Route::post('invitations/{token}/accept', [InvitationController::class, 'accept']);

// Protected routes
Route::middleware('auth:sanctum')->group(function () {
    // Auth
    Route::post('logout', [AuthController::class, 'logout']);
    Route::get('me', [AuthController::class, 'me']);
    
    // Profile
    Route::get('profile', [ProfileController::class, 'show']);
    Route::put('profile', [ProfileController::class, 'update']);
    Route::put('profile/password', [ProfileController::class, 'updatePassword']);
    Route::post('profile/avatar', [ProfileController::class, 'updateAvatar']);
    
    // Tenant
    Route::get('tenant', [TenantController::class, 'show']);
    Route::put('tenant', [TenantController::class, 'update']);
    Route::post('tenant/logo', [TenantController::class, 'updateLogo']);
    
    // Users
    Route::apiResource('users', UserController::class);
    Route::post('users/{user}/activate', [UserController::class, 'activate']);
    Route::post('users/{user}/deactivate', [UserController::class, 'deactivate']);
    
    // Teams
    Route::apiResource('teams', TeamController::class);
    Route::get('teams/{team}/members', [TeamController::class, 'members']);
    Route::post('teams/{team}/members', [TeamController::class, 'addMember']);
    Route::delete('teams/{team}/members/{user}', [TeamController::class, 'removeMember']);
    
    // Roles
    Route::apiResource('roles', RoleController::class);
    
    // Invitations
    Route::post('invitations', [InvitationController::class, 'store']);
    Route::delete('invitations/{invitation}', [InvitationController::class, 'destroy']);
    Route::post('invitations/{invitation}/resend', [InvitationController::class, 'resend']);
});
```

---

## ✅ COMPLETION CHECKLIST

- [ ] All 6 migrations created
- [ ] All 8 models with relationships
- [ ] All 7 controllers with CRUD
- [ ] All 5 services with business logic
- [ ] All 12 request validations
- [ ] All 3 middleware
- [ ] Routes file complete
- [ ] 2 seeders
- [ ] 5 feature tests

---

## 🔗 DEPENDENCIES
- None (this is the foundation)

## 🔗 OTHER BOTS DEPEND ON
- All other bots depend on Tenant, User models
- All other bots use BelongsToTenant trait

---

## 📤 OUTPUT
Save all files to: `/home/claude/business-platform/`
When done, create: `/home/claude/business-platform/docs/PROGRESS_BOT_1.md`
