<?php

namespace App\Services;

use App\Jobs\ProcessWorkflowStep;
use App\Models\WorkflowEnrollment;
use App\Models\WorkflowStep;
use Carbon\Carbon;

class WorkflowEngine
{
    public function __construct(
        protected EmailService $emailService,
        protected SmsService $smsService,
        protected WhatsappService $whatsappService
    ) {}

    public function processStep(WorkflowEnrollment $enrollment, int $stepIndex): void
    {
        if ($enrollment->status !== WorkflowEnrollment::STATUS_ACTIVE) {
            return;
        }

        $step = $enrollment->workflow->steps()->where('position', $stepIndex + 1)->first();

        if (!$step) {
            $enrollment->update(['status' => WorkflowEnrollment::STATUS_COMPLETED, 'completed_at' => now()]);
            return;
        }

        // Handle delay
        if ($step->delay && !$enrollment->delay_until) {
            $delayUntil = $this->calculateDelay($step->delay);
            $enrollment->update(['delay_until' => $delayUntil, 'current_step' => $stepIndex]);
            ProcessWorkflowStep::dispatch($enrollment, $stepIndex)->delay($delayUntil);
            return;
        }

        $enrollment->update(['delay_until' => null]);

        try {
            $this->executeStep($step, $enrollment);
            $enrollment->update(['current_step' => $stepIndex + 1]);
            ProcessWorkflowStep::dispatch($enrollment, $stepIndex + 1);
        } catch (\Exception $e) {
            $enrollment->update([
                'status' => WorkflowEnrollment::STATUS_FAILED,
                'error' => $e->getMessage(),
            ]);
        }
    }

    protected function calculateDelay(array $delay): Carbon
    {
        $value = $delay['value'] ?? 1;
        $unit = $delay['unit'] ?? 'days';

        return match ($unit) {
            'minutes' => now()->addMinutes($value),
            'hours' => now()->addHours($value),
            'days' => now()->addDays($value),
            'weeks' => now()->addWeeks($value),
            default => now()->addDays($value),
        };
    }

    protected function executeStep(WorkflowStep $step, WorkflowEnrollment $enrollment): void
    {
        $contact = $enrollment->contact;
        $config = $step->config;

        match ($step->type) {
            'send_email' => $this->sendEmail($contact, $config),
            'send_sms' => $this->sendSms($contact, $config),
            'send_whatsapp' => $this->sendWhatsapp($contact, $config),
            'add_tag' => $this->addTag($contact, $config),
            'remove_tag' => $this->removeTag($contact, $config),
            'update_field' => $this->updateField($contact, $config),
            'create_task' => $this->createTask($contact, $config, $enrollment),
            'webhook' => $this->callWebhook($contact, $config, $enrollment),
            'if_else' => $this->handleCondition($contact, $config, $enrollment),
            default => null,
        };
    }

    protected function sendEmail($contact, array $config): void
    {
        $this->emailService->sendFromTemplate(
            $contact,
            $config['template_id'],
            $config['variables'] ?? []
        );
    }

    protected function sendSms($contact, array $config): void
    {
        $this->smsService->send($contact->phone, $config['message']);
    }

    protected function sendWhatsapp($contact, array $config): void
    {
        $this->whatsappService->sendTemplate($contact->phone, $config['template'], $config['params'] ?? []);
    }

    protected function addTag($contact, array $config): void
    {
        $contact->tags()->syncWithoutDetaching($config['tag_ids']);
    }

    protected function removeTag($contact, array $config): void
    {
        $contact->tags()->detach($config['tag_ids']);
    }

    protected function updateField($contact, array $config): void
    {
        $contact->update([$config['field'] => $config['value']]);
    }

    protected function createTask($contact, array $config, WorkflowEnrollment $enrollment): void
    {
        \App\Models\Task::create([
            'tenant_id' => $contact->tenant_id,
            'project_id' => $config['project_id'],
            'title' => $this->replaceVariables($config['title'], $contact),
            'description' => $this->replaceVariables($config['description'] ?? '', $contact),
            'due_date' => now()->addDays($config['due_days'] ?? 7),
            'created_by' => $enrollment->workflow->created_by,
        ]);
    }

    protected function callWebhook($contact, array $config, WorkflowEnrollment $enrollment): void
    {
        \Illuminate\Support\Facades\Http::post($config['url'], [
            'contact' => $contact->toArray(),
            'enrollment_id' => $enrollment->id,
            'workflow_id' => $enrollment->workflow_id,
        ]);
    }

    protected function handleCondition($contact, array $config, WorkflowEnrollment $enrollment): void
    {
        // Branch logic handled separately
    }

    protected function replaceVariables(string $text, $contact): string
    {
        return str_replace(
            ['{{first_name}}', '{{last_name}}', '{{email}}', '{{company}}'],
            [$contact->first_name, $contact->last_name, $contact->email, $contact->company_name],
            $text
        );
    }
}
