<?php

namespace App\Services;

use App\Models\Campaign;
use App\Models\CampaignRecipient;
use App\Models\WhatsappMessage;
use Illuminate\Support\Facades\Http;

class WhatsappService
{
    protected string $apiUrl;
    protected string $apiToken;
    protected string $phoneNumberId;

    public function __construct()
    {
        $this->apiUrl = 'https://graph.facebook.com/v17.0';
        $this->apiToken = config('services.whatsapp.token');
        $this->phoneNumberId = config('services.whatsapp.phone_number_id');
    }

    public function send(int $tenantId, array $data, int $userId): WhatsappMessage
    {
        $message = WhatsappMessage::create([
            'tenant_id' => $tenantId,
            'contact_id' => $data['contact_id'] ?? null,
            'direction' => 'outbound',
            'from_number' => $this->phoneNumberId,
            'to_number' => $this->formatPhoneNumber($data['to']),
            'message_type' => $data['message_type'],
            'content' => $data['content'] ?? null,
            'template_name' => $data['template_name'] ?? null,
            'template_params' => $data['template_params'] ?? null,
            'media_url' => $data['media_url'] ?? null,
            'created_by' => $userId,
        ]);

        $this->dispatch($message);

        return $message;
    }

    protected function dispatch(WhatsappMessage $message): void
    {
        if (!$this->apiToken) {
            $message->update(['status' => WhatsappMessage::STATUS_FAILED, 'error' => 'WhatsApp not configured']);
            return;
        }

        try {
            $payload = $this->buildPayload($message);

            $response = Http::withToken($this->apiToken)
                ->post("{$this->apiUrl}/{$this->phoneNumberId}/messages", $payload);

            if ($response->successful()) {
                $data = $response->json();
                $message->update([
                    'external_id' => $data['messages'][0]['id'] ?? null,
                    'status' => WhatsappMessage::STATUS_SENT,
                    'sent_at' => now(),
                ]);
            } else {
                $message->update([
                    'status' => WhatsappMessage::STATUS_FAILED,
                    'error' => $response->json('error.message') ?? 'Unknown error',
                ]);
            }
        } catch (\Exception $e) {
            $message->update([
                'status' => WhatsappMessage::STATUS_FAILED,
                'error' => $e->getMessage(),
            ]);
        }
    }

    protected function buildPayload(WhatsappMessage $message): array
    {
        $payload = [
            'messaging_product' => 'whatsapp',
            'recipient_type' => 'individual',
            'to' => $message->to_number,
        ];

        return match ($message->message_type) {
            'text' => array_merge($payload, [
                'type' => 'text',
                'text' => ['body' => $message->content],
            ]),
            'template' => array_merge($payload, [
                'type' => 'template',
                'template' => [
                    'name' => $message->template_name,
                    'language' => ['code' => 'en'],
                    'components' => $this->buildTemplateComponents($message->template_params),
                ],
            ]),
            'image' => array_merge($payload, [
                'type' => 'image',
                'image' => ['link' => $message->media_url],
            ]),
            'document' => array_merge($payload, [
                'type' => 'document',
                'document' => ['link' => $message->media_url],
            ]),
            default => $payload,
        };
    }

    protected function buildTemplateComponents(?array $params): array
    {
        if (!$params) return [];

        return [[
            'type' => 'body',
            'parameters' => array_map(fn($value) => [
                'type' => 'text',
                'text' => $value,
            ], $params),
        ]];
    }

    public function sendCampaignWhatsapp(Campaign $campaign, CampaignRecipient $recipient): void
    {
        $contact = $recipient->contact;
        $phone = $contact->mobile ?? $contact->phone;

        if (!$phone) {
            throw new \Exception('Contact has no phone number');
        }

        $this->send($campaign->tenant_id, [
            'to' => $phone,
            'message_type' => 'template',
            'template_name' => $campaign->settings['template_name'] ?? 'marketing_message',
            'template_params' => [
                $contact->first_name,
                $campaign->name,
            ],
            'contact_id' => $contact->id,
        ], $campaign->created_by);
    }

    protected function formatPhoneNumber(string $number): string
    {
        $number = preg_replace('/[^0-9]/', '', $number);
        if (strlen($number) === 8) {
            $number = '973' . $number; // Bahrain
        }
        return $number;
    }

    public function handleWebhook(array $data): void
    {
        $entry = $data['entry'][0] ?? null;
        if (!$entry) return;

        $changes = $entry['changes'][0] ?? null;
        if (!$changes) return;

        $value = $changes['value'] ?? null;

        // Handle incoming messages
        if (isset($value['messages'])) {
            foreach ($value['messages'] as $msg) {
                $this->handleIncomingMessage($msg, $value['contacts'][0] ?? null);
            }
        }

        // Handle status updates
        if (isset($value['statuses'])) {
            foreach ($value['statuses'] as $status) {
                $this->handleStatusUpdate($status);
            }
        }
    }

    protected function handleIncomingMessage(array $msg, ?array $contact): void
    {
        // Find or create contact and store message
        // Implementation depends on business logic
    }

    protected function handleStatusUpdate(array $status): void
    {
        $message = WhatsappMessage::where('external_id', $status['id'])->first();
        if (!$message) return;

        $statusMap = [
            'sent' => WhatsappMessage::STATUS_SENT,
            'delivered' => WhatsappMessage::STATUS_DELIVERED,
            'read' => WhatsappMessage::STATUS_READ,
            'failed' => WhatsappMessage::STATUS_FAILED,
        ];

        if (isset($statusMap[$status['status']])) {
            $message->update([
                'status' => $statusMap[$status['status']],
                'delivered_at' => $status['status'] === 'delivered' ? now() : $message->delivered_at,
                'read_at' => $status['status'] === 'read' ? now() : $message->read_at,
            ]);
        }
    }
}
