<?php

namespace App\Services;

use App\Models\SmsMessage;
use Twilio\Rest\Client;

class TwilioService
{
    protected Client $client;
    protected string $fromNumber;

    public function __construct()
    {
        $this->client = new Client(
            config('services.twilio.sid'),
            config('services.twilio.token')
        );
        $this->fromNumber = config('services.twilio.from');
    }

    public function send(string $to, string $message, ?int $tenantId = null, ?int $contactId = null): SmsMessage
    {
        $smsMessage = SmsMessage::create([
            'tenant_id' => $tenantId,
            'contact_id' => $contactId,
            'direction' => 'outbound',
            'from_number' => $this->fromNumber,
            'to_number' => $to,
            'content' => $message,
            'status' => SmsMessage::STATUS_QUEUED,
        ]);

        try {
            $twilioMessage = $this->client->messages->create($to, [
                'from' => $this->fromNumber,
                'body' => $message,
            ]);

            $smsMessage->update([
                'external_id' => $twilioMessage->sid,
                'status' => SmsMessage::STATUS_SENT,
                'sent_at' => now(),
            ]);
        } catch (\Exception $e) {
            $smsMessage->update([
                'status' => SmsMessage::STATUS_FAILED,
                'error' => $e->getMessage(),
            ]);
        }

        return $smsMessage;
    }

    public function handleWebhook(array $data): void
    {
        $message = SmsMessage::where('external_id', $data['MessageSid'])->first();

        if ($message) {
            $status = match ($data['MessageStatus']) {
                'delivered' => SmsMessage::STATUS_DELIVERED,
                'failed', 'undelivered' => SmsMessage::STATUS_FAILED,
                default => $message->status,
            };

            $message->update([
                'status' => $status,
                'delivered_at' => $status === SmsMessage::STATUS_DELIVERED ? now() : null,
                'error' => $data['ErrorMessage'] ?? null,
            ]);
        }
    }

    public function handleIncoming(array $data): SmsMessage
    {
        return SmsMessage::create([
            'direction' => 'inbound',
            'external_id' => $data['MessageSid'],
            'from_number' => $data['From'],
            'to_number' => $data['To'],
            'content' => $data['Body'],
            'status' => SmsMessage::STATUS_RECEIVED,
            'received_at' => now(),
        ]);
    }
}
