<?php

namespace App\Services;

use App\Models\Task;
use App\Models\TimeEntry;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class TimeEntryService
{
    public function start(int $tenantId, int $userId, int $taskId, ?string $description = null): TimeEntry
    {
        // Stop any running timers
        $this->stopAllRunning($tenantId, $userId);

        return TimeEntry::create([
            'tenant_id' => $tenantId,
            'user_id' => $userId,
            'task_id' => $taskId,
            'description' => $description,
            'start_time' => now(),
            'is_running' => true,
        ]);
    }

    public function stop(TimeEntry $entry): TimeEntry
    {
        $endTime = now();
        $duration = $entry->start_time->diffInMinutes($endTime);

        $entry->update([
            'end_time' => $endTime,
            'duration' => $duration,
            'is_running' => false,
        ]);

        // Update task logged hours
        $this->updateTaskLoggedHours($entry->task);

        return $entry;
    }

    public function manual(int $tenantId, int $userId, array $data): TimeEntry
    {
        $startTime = Carbon::parse($data['start_time']);
        $endTime = Carbon::parse($data['end_time']);
        $duration = $startTime->diffInMinutes($endTime);

        $entry = TimeEntry::create([
            'tenant_id' => $tenantId,
            'user_id' => $userId,
            'task_id' => $data['task_id'],
            'description' => $data['description'] ?? null,
            'start_time' => $startTime,
            'end_time' => $endTime,
            'duration' => $duration,
            'is_billable' => $data['is_billable'] ?? true,
            'is_running' => false,
        ]);

        $this->updateTaskLoggedHours($entry->task);

        return $entry;
    }

    public function stopAllRunning(int $tenantId, int $userId): void
    {
        $runningEntries = TimeEntry::where('tenant_id', $tenantId)
            ->where('user_id', $userId)
            ->where('is_running', true)
            ->get();

        foreach ($runningEntries as $entry) {
            $this->stop($entry);
        }
    }

    public function getRunning(int $tenantId, int $userId): ?TimeEntry
    {
        return TimeEntry::where('tenant_id', $tenantId)
            ->where('user_id', $userId)
            ->where('is_running', true)
            ->first();
    }

    public function getUserSummary(int $tenantId, int $userId, string $startDate, string $endDate): array
    {
        $entries = TimeEntry::where('tenant_id', $tenantId)
            ->where('user_id', $userId)
            ->whereBetween('start_time', [$startDate, $endDate])
            ->where('is_running', false)
            ->get();

        return [
            'total_hours' => round($entries->sum('duration') / 60, 2),
            'billable_hours' => round($entries->where('is_billable', true)->sum('duration') / 60, 2),
            'entries_count' => $entries->count(),
            'by_project' => $entries->groupBy('task.project_id')
                ->map(fn($group) => round($group->sum('duration') / 60, 2)),
        ];
    }

    protected function updateTaskLoggedHours(Task $task): void
    {
        $totalMinutes = $task->timeEntries()->sum('duration');
        $task->update(['logged_hours' => round($totalMinutes / 60, 2)]);
    }
}
