<?php

namespace App\Services;

use App\Models\Campaign;
use App\Models\CampaignRecipient;
use App\Models\SmsMessage;
use Twilio\Rest\Client as TwilioClient;

class SmsService
{
    protected ?TwilioClient $twilioClient = null;

    public function __construct()
    {
        if (config('services.twilio.sid') && config('services.twilio.token')) {
            $this->twilioClient = new TwilioClient(
                config('services.twilio.sid'),
                config('services.twilio.token')
            );
        }
    }

    public function send(int $tenantId, array $data, int $userId): SmsMessage
    {
        $message = SmsMessage::create([
            'tenant_id' => $tenantId,
            'contact_id' => $data['contact_id'] ?? null,
            'direction' => 'outbound',
            'from_number' => $data['from_number'] ?? config('services.twilio.from'),
            'to_number' => $this->formatPhoneNumber($data['to']),
            'content' => $data['message'],
            'segments' => $this->calculateSegments($data['message']),
            'created_by' => $userId,
        ]);

        $this->dispatch($message);

        return $message;
    }

    protected function dispatch(SmsMessage $message): void
    {
        if (!$this->twilioClient) {
            $message->update(['status' => SmsMessage::STATUS_FAILED, 'error' => 'Twilio not configured']);
            return;
        }

        try {
            $result = $this->twilioClient->messages->create(
                $message->to_number,
                [
                    'from' => $message->from_number,
                    'body' => $message->content,
                    'statusCallback' => route('webhooks.twilio.status'),
                ]
            );

            $message->update([
                'external_id' => $result->sid,
                'status' => SmsMessage::STATUS_SENT,
                'sent_at' => now(),
            ]);
        } catch (\Exception $e) {
            $message->update([
                'status' => SmsMessage::STATUS_FAILED,
                'error' => $e->getMessage(),
            ]);
        }
    }

    public function sendCampaignSms(Campaign $campaign, CampaignRecipient $recipient): void
    {
        $contact = $recipient->contact;
        if (!$contact->phone && !$contact->mobile) {
            throw new \Exception('Contact has no phone number');
        }

        $content = $this->replaceVariables($campaign->content, $contact);

        $this->send($campaign->tenant_id, [
            'to' => $contact->mobile ?? $contact->phone,
            'message' => $content,
            'contact_id' => $contact->id,
        ], $campaign->created_by);
    }

    protected function replaceVariables(string $content, $contact): string
    {
        return str_replace(
            ['{{first_name}}', '{{last_name}}', '{{company}}'],
            [$contact->first_name, $contact->last_name, $contact->company_name],
            $content
        );
    }

    protected function formatPhoneNumber(string $number): string
    {
        $number = preg_replace('/[^0-9+]/', '', $number);
        if (!str_starts_with($number, '+')) {
            $number = '+973' . $number; // Default to Bahrain
        }
        return $number;
    }

    protected function calculateSegments(string $message): int
    {
        $length = mb_strlen($message);
        if ($length <= 160) return 1;
        return (int) ceil($length / 153);
    }

    public function handleStatusCallback(array $data): void
    {
        $message = SmsMessage::where('external_id', $data['MessageSid'])->first();
        if (!$message) return;

        $statusMap = [
            'delivered' => SmsMessage::STATUS_DELIVERED,
            'failed' => SmsMessage::STATUS_FAILED,
            'undelivered' => SmsMessage::STATUS_FAILED,
        ];

        if (isset($statusMap[$data['MessageStatus']])) {
            $message->update([
                'status' => $statusMap[$data['MessageStatus']],
                'delivered_at' => $data['MessageStatus'] === 'delivered' ? now() : null,
                'error' => $data['ErrorMessage'] ?? null,
            ]);
        }
    }
}
