<?php

namespace App\Services;

use App\Models\Bill;
use App\Models\ChartOfAccount;
use App\Models\Contact;
use App\Models\Expense;
use App\Models\Invoice;
use App\Models\JournalEntry;
use App\Models\Opportunity;
use App\Models\Task;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ReportService
{
    public function generateBalanceSheet(int $tenantId, string $asOfDate): array
    {
        $accounts = ChartOfAccount::where('tenant_id', $tenantId)
            ->with(['journalEntryLines' => function ($query) use ($asOfDate) {
                $query->whereHas('journalEntry', fn($q) => $q->where('date', '<=', $asOfDate));
            }])
            ->get();

        $groupedAccounts = $accounts->groupBy('type');

        return [
            'as_of_date' => $asOfDate,
            'assets' => $this->formatAccountGroup($groupedAccounts->get('asset', collect()), 'debit'),
            'liabilities' => $this->formatAccountGroup($groupedAccounts->get('liability', collect()), 'credit'),
            'equity' => $this->formatAccountGroup($groupedAccounts->get('equity', collect()), 'credit'),
        ];
    }

    public function generateProfitAndLoss(int $tenantId, string $startDate, string $endDate): array
    {
        $revenue = Invoice::where('tenant_id', $tenantId)
            ->whereBetween('invoice_date', [$startDate, $endDate])
            ->where('status', 'paid')
            ->sum('total');

        $expenses = Expense::where('tenant_id', $tenantId)
            ->whereBetween('expense_date', [$startDate, $endDate])
            ->where('status', 'approved')
            ->sum('amount');

        $billExpenses = Bill::where('tenant_id', $tenantId)
            ->whereBetween('bill_date', [$startDate, $endDate])
            ->sum('total');

        return [
            'period' => ['start' => $startDate, 'end' => $endDate],
            'revenue' => [
                'items' => [['name' => 'Sales', 'amount' => $revenue]],
                'total' => $revenue,
            ],
            'expenses' => [
                'items' => [
                    ['name' => 'Operating Expenses', 'amount' => $expenses],
                    ['name' => 'Cost of Goods', 'amount' => $billExpenses],
                ],
                'total' => $expenses + $billExpenses,
            ],
            'net_income' => $revenue - $expenses - $billExpenses,
        ];
    }

    public function generateCashFlow(int $tenantId, string $period): array
    {
        [$startDate, $endDate] = $this->getPeriodDates($period);

        $operatingInflows = Invoice::where('tenant_id', $tenantId)
            ->whereBetween('paid_at', [$startDate, $endDate])
            ->sum('amount_paid');

        $operatingOutflows = Bill::where('tenant_id', $tenantId)
            ->whereBetween('paid_at', [$startDate, $endDate])
            ->sum('amount_paid');

        $expenseOutflows = Expense::where('tenant_id', $tenantId)
            ->whereBetween('expense_date', [$startDate, $endDate])
            ->where('status', 'approved')
            ->sum('amount');

        return [
            'period' => ['start' => $startDate, 'end' => $endDate],
            'operating' => [
                ['label' => 'Collections from Customers', 'amount' => $operatingInflows],
                ['label' => 'Payments to Vendors', 'amount' => -$operatingOutflows],
                ['label' => 'Operating Expenses', 'amount' => -$expenseOutflows],
            ],
            'operating_total' => $operatingInflows - $operatingOutflows - $expenseOutflows,
            'investing' => [],
            'investing_total' => 0,
            'financing' => [],
            'financing_total' => 0,
            'net_change' => $operatingInflows - $operatingOutflows - $expenseOutflows,
            'beginning_balance' => 0,
            'ending_balance' => $operatingInflows - $operatingOutflows - $expenseOutflows,
        ];
    }

    public function generateAgingReceivables(int $tenantId): array
    {
        $invoices = Invoice::where('tenant_id', $tenantId)
            ->whereIn('status', ['sent', 'partial', 'overdue'])
            ->with('contact')
            ->get();

        return $this->calculateAging($invoices, 'contact', 'due_date', 'total', 'amount_paid');
    }

    public function generateAgingPayables(int $tenantId): array
    {
        $bills = Bill::where('tenant_id', $tenantId)
            ->whereIn('status', ['received', 'partial'])
            ->with('vendor')
            ->get();

        return $this->calculateAging($bills, 'vendor', 'due_date', 'total', 'amount_paid');
    }

    public function generateSalesReport(int $tenantId, string $startDate, string $endDate): array
    {
        $invoices = Invoice::where('tenant_id', $tenantId)
            ->whereBetween('invoice_date', [$startDate, $endDate])
            ->with('contact')
            ->get();

        $totalSales = $invoices->sum('total');
        $topCustomer = $invoices->groupBy('contact_id')
            ->map(fn($group) => $group->sum('total'))
            ->sortDesc()
            ->keys()
            ->first();

        return [
            'stats' => [
                'total_sales' => $totalSales,
                'invoices_count' => $invoices->count(),
                'average_sale' => $invoices->count() ? $totalSales / $invoices->count() : 0,
                'top_customer' => Contact::find($topCustomer)?->company_name ?? '-',
            ],
            'invoices' => $invoices,
            'top_products' => [],
        ];
    }

    public function getRevenueForPeriod(int $tenantId, string $startDate, string $endDate): float
    {
        return Invoice::where('tenant_id', $tenantId)
            ->whereBetween('invoice_date', [$startDate, $endDate])
            ->where('status', 'paid')
            ->sum('total');
    }

    public function getExpensesForPeriod(int $tenantId, string $startDate, string $endDate): float
    {
        return Expense::where('tenant_id', $tenantId)
            ->whereBetween('expense_date', [$startDate, $endDate])
            ->where('status', 'approved')
            ->sum('amount');
    }

    public function getNewContactsForPeriod(int $tenantId, string $startDate, string $endDate): int
    {
        return Contact::where('tenant_id', $tenantId)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->count();
    }

    public function getWonOpportunitiesForPeriod(int $tenantId, string $startDate, string $endDate): float
    {
        return Opportunity::where('tenant_id', $tenantId)
            ->where('status', 'won')
            ->whereBetween('closed_at', [$startDate, $endDate])
            ->sum('value');
    }

    public function getCompletedTasksForPeriod(int $tenantId, string $startDate, string $endDate): int
    {
        return Task::where('tenant_id', $tenantId)
            ->whereNotNull('completed_at')
            ->whereBetween('completed_at', [$startDate, $endDate])
            ->count();
    }

    protected function formatAccountGroup($accounts, string $normalBalance): array
    {
        $items = $accounts->map(function ($account) use ($normalBalance) {
            $debits = $account->journalEntryLines->sum('debit');
            $credits = $account->journalEntryLines->sum('credit');
            $balance = $normalBalance === 'debit' ? $debits - $credits : $credits - $debits;

            return ['name' => $account->name, 'balance' => $balance];
        })->filter(fn($item) => $item['balance'] != 0);

        return [
            'items' => $items->values(),
            'total' => $items->sum('balance'),
        ];
    }

    protected function getPeriodDates(string $period): array
    {
        return match ($period) {
            'this_month' => [now()->startOfMonth()->toDateString(), now()->toDateString()],
            'last_month' => [now()->subMonth()->startOfMonth()->toDateString(), now()->subMonth()->endOfMonth()->toDateString()],
            'this_quarter' => [now()->startOfQuarter()->toDateString(), now()->toDateString()],
            'this_year' => [now()->startOfYear()->toDateString(), now()->toDateString()],
            default => [now()->startOfMonth()->toDateString(), now()->toDateString()],
        };
    }

    protected function calculateAging($items, string $entityRelation, string $dateField, string $totalField, string $paidField): array
    {
        $today = now();
        $grouped = [];
        $totals = ['current' => 0, 'days_1_30' => 0, 'days_31_60' => 0, 'days_61_90' => 0, 'days_90_plus' => 0];

        foreach ($items as $item) {
            $entity = $item->$entityRelation;
            $entityId = $entity->id;
            $entityName = $entity->name ?? $entity->company_name ?? 'Unknown';
            $outstanding = $item->$totalField - $item->$paidField;
            $daysPastDue = $today->diffInDays(Carbon::parse($item->$dateField), false);

            if (!isset($grouped[$entityId])) {
                $grouped[$entityId] = [
                    'id' => $entityId,
                    'name' => $entityName,
                    'current' => 0, 'days_1_30' => 0, 'days_31_60' => 0, 'days_61_90' => 0, 'days_90_plus' => 0, 'total' => 0,
                ];
            }

            $bucket = match (true) {
                $daysPastDue <= 0 => 'current',
                $daysPastDue <= 30 => 'days_1_30',
                $daysPastDue <= 60 => 'days_31_60',
                $daysPastDue <= 90 => 'days_61_90',
                default => 'days_90_plus',
            };

            $grouped[$entityId][$bucket] += $outstanding;
            $grouped[$entityId]['total'] += $outstanding;
            $totals[$bucket] += $outstanding;
        }

        return ['items' => array_values($grouped), 'totals' => $totals];
    }
}
