<?php

namespace App\Services;

use App\Models\BankAccount;
use App\Models\BankTransaction;
use Illuminate\Support\Facades\Http;

class PlaidService
{
    protected string $baseUrl;
    protected string $clientId;
    protected string $secret;

    public function __construct()
    {
        $this->baseUrl = config('services.plaid.environment') === 'production'
            ? 'https://production.plaid.com'
            : 'https://sandbox.plaid.com';
        $this->clientId = config('services.plaid.client_id');
        $this->secret = config('services.plaid.secret');
    }

    public function createLinkToken(int $userId): string
    {
        $response = Http::post("{$this->baseUrl}/link/token/create", [
            'client_id' => $this->clientId,
            'secret' => $this->secret,
            'user' => ['client_user_id' => (string) $userId],
            'client_name' => config('app.name'),
            'products' => ['transactions'],
            'country_codes' => ['US', 'BH'],
            'language' => 'en',
        ]);

        return $response->json('link_token');
    }

    public function exchangePublicToken(string $publicToken): array
    {
        $response = Http::post("{$this->baseUrl}/item/public_token/exchange", [
            'client_id' => $this->clientId,
            'secret' => $this->secret,
            'public_token' => $publicToken,
        ]);

        return [
            'access_token' => $response->json('access_token'),
            'item_id' => $response->json('item_id'),
        ];
    }

    public function getAccounts(string $accessToken): array
    {
        $response = Http::post("{$this->baseUrl}/accounts/get", [
            'client_id' => $this->clientId,
            'secret' => $this->secret,
            'access_token' => $accessToken,
        ]);

        return $response->json('accounts', []);
    }

    public function syncTransactions(BankAccount $bankAccount): void
    {
        $accessToken = $bankAccount->plaid_access_token;
        $cursor = $bankAccount->plaid_cursor;

        $hasMore = true;

        while ($hasMore) {
            $response = Http::post("{$this->baseUrl}/transactions/sync", [
                'client_id' => $this->clientId,
                'secret' => $this->secret,
                'access_token' => $accessToken,
                'cursor' => $cursor,
            ]);

            $data = $response->json();

            // Process added transactions
            foreach ($data['added'] ?? [] as $transaction) {
                $this->createTransaction($bankAccount, $transaction);
            }

            // Process modified transactions
            foreach ($data['modified'] ?? [] as $transaction) {
                $this->updateTransaction($bankAccount, $transaction);
            }

            // Process removed transactions
            foreach ($data['removed'] ?? [] as $removed) {
                BankTransaction::where('bank_account_id', $bankAccount->id)
                    ->where('provider_id', $removed['transaction_id'])
                    ->delete();
            }

            $cursor = $data['next_cursor'];
            $hasMore = $data['has_more'];
        }

        $bankAccount->update(['plaid_cursor' => $cursor]);
    }

    protected function createTransaction(BankAccount $bankAccount, array $data): BankTransaction
    {
        return BankTransaction::updateOrCreate(
            [
                'bank_account_id' => $bankAccount->id,
                'provider_id' => $data['transaction_id'],
            ],
            [
                'tenant_id' => $bankAccount->tenant_id,
                'transaction_date' => $data['date'],
                'description' => $data['name'],
                'amount' => abs($data['amount']),
                'type' => $data['amount'] > 0 ? 'debit' : 'credit',
                'category' => $data['category'][0] ?? null,
                'subcategory' => $data['category'][1] ?? null,
                'merchant_name' => $data['merchant_name'] ?? null,
                'pending' => $data['pending'],
                'raw_data' => $data,
            ]
        );
    }

    protected function updateTransaction(BankAccount $bankAccount, array $data): void
    {
        BankTransaction::where('bank_account_id', $bankAccount->id)
            ->where('provider_id', $data['transaction_id'])
            ->update([
                'transaction_date' => $data['date'],
                'description' => $data['name'],
                'amount' => abs($data['amount']),
                'type' => $data['amount'] > 0 ? 'debit' : 'credit',
                'category' => $data['category'][0] ?? null,
                'merchant_name' => $data['merchant_name'] ?? null,
                'pending' => $data['pending'],
                'raw_data' => $data,
            ]);
    }

    public function getBalance(string $accessToken): array
    {
        $response = Http::post("{$this->baseUrl}/accounts/balance/get", [
            'client_id' => $this->clientId,
            'secret' => $this->secret,
            'access_token' => $accessToken,
        ]);

        return $response->json('accounts', []);
    }
}
