<?php

namespace App\Services;

use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Payment;
use App\Models\JournalEntry;
use App\Models\ChartOfAccount;
use Illuminate\Support\Facades\DB;

class InvoiceService
{
    public function create(array $data): Invoice
    {
        return DB::transaction(function () use ($data) {
            $invoice = Invoice::create([
                'tenant_id' => auth()->user()->tenant_id,
                'contact_id' => $data['contact_id'],
                'opportunity_id' => $data['opportunity_id'] ?? null,
                'invoice_number' => $this->generateInvoiceNumber(),
                'invoice_date' => $data['invoice_date'],
                'due_date' => $data['due_date'],
                'currency' => $data['currency'] ?? 'BHD',
                'notes' => $data['notes'] ?? null,
                'terms' => $data['terms'] ?? null,
                'created_by' => auth()->id(),
            ]);

            $this->syncItems($invoice, $data['items']);
            $this->calculateTotals($invoice);

            return $invoice;
        });
    }

    public function update(Invoice $invoice, array $data): Invoice
    {
        return DB::transaction(function () use ($invoice, $data) {
            $invoice->update(collect($data)->except(['items'])->toArray());

            if (isset($data['items'])) {
                $this->syncItems($invoice, $data['items']);
            }

            $this->calculateTotals($invoice);
            return $invoice->fresh();
        });
    }

    protected function syncItems(Invoice $invoice, array $items): void
    {
        $existingIds = [];

        foreach ($items as $index => $itemData) {
            if (isset($itemData['id'])) {
                $item = InvoiceItem::find($itemData['id']);
                $item->update($this->prepareItemData($itemData, $index));
                $existingIds[] = $item->id;
            } else {
                $item = $invoice->items()->create($this->prepareItemData($itemData, $index));
                $existingIds[] = $item->id;
            }
        }

        $invoice->items()->whereNotIn('id', $existingIds)->delete();
    }

    protected function prepareItemData(array $data, int $position): array
    {
        $quantity = $data['quantity'] ?? 1;
        $unitPrice = $data['unit_price'] ?? 0;
        $discount = $data['discount'] ?? 0;
        $taxRate = $data['tax_rate'] ?? 10;

        $subtotal = $quantity * $unitPrice;
        $discountAmount = $subtotal * ($discount / 100);
        $taxableAmount = $subtotal - $discountAmount;
        $taxAmount = $taxableAmount * ($taxRate / 100);
        $total = $taxableAmount + $taxAmount;

        return [
            'product_id' => $data['product_id'] ?? null,
            'name' => $data['name'],
            'description' => $data['description'] ?? null,
            'quantity' => $quantity,
            'unit_price' => $unitPrice,
            'discount' => $discount,
            'tax_rate' => $taxRate,
            'tax_amount' => $taxAmount,
            'total' => $total,
            'position' => $position,
        ];
    }

    public function calculateTotals(Invoice $invoice): void
    {
        $subtotal = $invoice->items()->sum(DB::raw('quantity * unit_price'));
        $discountAmount = $invoice->items()->sum(DB::raw('quantity * unit_price * discount / 100'));
        $taxAmount = $invoice->items()->sum('tax_amount');
        $total = $subtotal - $discountAmount + $taxAmount;

        $invoice->update([
            'subtotal' => $subtotal,
            'discount_amount' => $discountAmount,
            'tax_amount' => $taxAmount,
            'total' => $total,
            'amount_due' => $total - $invoice->amount_paid,
        ]);
    }

    public function send(Invoice $invoice): void
    {
        $invoice->update([
            'status' => 'sent',
            'sent_at' => now(),
        ]);

        // Send email notification
        // Mail::to($invoice->contact->email)->send(new InvoiceMail($invoice));
    }

    public function recordPayment(Invoice $invoice, array $data): Payment
    {
        return DB::transaction(function () use ($invoice, $data) {
            $payment = Payment::create([
                'tenant_id' => $invoice->tenant_id,
                'contact_id' => $invoice->contact_id,
                'invoice_id' => $invoice->id,
                'payment_number' => $this->generatePaymentNumber(),
                'payment_date' => $data['payment_date'],
                'amount' => $data['amount'],
                'currency' => $invoice->currency,
                'method' => $data['method'],
                'reference' => $data['reference'] ?? null,
                'notes' => $data['notes'] ?? null,
                'bank_account_id' => $data['bank_account_id'] ?? null,
                'created_by' => auth()->id(),
            ]);

            $invoice->amount_paid += $data['amount'];
            $invoice->amount_due = $invoice->total - $invoice->amount_paid;

            if ($invoice->amount_due <= 0) {
                $invoice->status = 'paid';
                $invoice->paid_at = now();
            } else {
                $invoice->status = 'partial';
            }

            $invoice->save();

            $this->createPaymentJournalEntry($payment);

            return $payment;
        });
    }

    protected function createPaymentJournalEntry(Payment $payment): void
    {
        $journalEntry = JournalEntry::create([
            'tenant_id' => $payment->tenant_id,
            'entry_number' => 'JE-' . str_pad(JournalEntry::count() + 1, 6, '0', STR_PAD_LEFT),
            'entry_date' => $payment->payment_date,
            'type' => 'payment',
            'journalable_type' => Payment::class,
            'journalable_id' => $payment->id,
            'description' => "Payment received for Invoice #{$payment->invoice->invoice_number}",
            'status' => 'posted',
            'created_by' => auth()->id(),
        ]);

        // Debit: Bank/Cash
        $journalEntry->lines()->create([
            'account_id' => $this->getBankAccount($payment)->id,
            'debit' => $payment->amount,
            'credit' => 0,
            'description' => 'Payment received',
        ]);

        // Credit: Accounts Receivable
        $journalEntry->lines()->create([
            'account_id' => $this->getAccountsReceivable()->id,
            'debit' => 0,
            'credit' => $payment->amount,
            'contact_id' => $payment->contact_id,
            'description' => 'Payment received',
        ]);

        $journalEntry->update([
            'total_debit' => $payment->amount,
            'total_credit' => $payment->amount,
        ]);
    }

    protected function getBankAccount(Payment $payment): ChartOfAccount
    {
        if ($payment->bank_account_id) {
            return ChartOfAccount::find($payment->bankAccount->account_id);
        }

        return ChartOfAccount::where('type', 'asset')
            ->where('is_bank_account', true)
            ->first() ?? ChartOfAccount::where('code', '1000')->first();
    }

    protected function getAccountsReceivable(): ChartOfAccount
    {
        return ChartOfAccount::where('code', '1100')->first()
            ?? ChartOfAccount::where('name', 'like', '%Accounts Receivable%')->first();
    }

    public function duplicate(Invoice $invoice): Invoice
    {
        $newInvoice = $invoice->replicate(['invoice_number', 'status', 'sent_at', 'viewed_at', 'paid_at', 'amount_paid']);
        $newInvoice->invoice_number = $this->generateInvoiceNumber();
        $newInvoice->invoice_date = now();
        $newInvoice->due_date = now()->addDays(30);
        $newInvoice->status = 'draft';
        $newInvoice->amount_paid = 0;
        $newInvoice->amount_due = $invoice->total;
        $newInvoice->save();

        foreach ($invoice->items as $item) {
            $newItem = $item->replicate();
            $newItem->invoice_id = $newInvoice->id;
            $newItem->save();
        }

        return $newInvoice;
    }

    public function void(Invoice $invoice): void
    {
        DB::transaction(function () use ($invoice) {
            // Reverse any journal entries
            JournalEntry::where('journalable_type', Invoice::class)
                ->where('journalable_id', $invoice->id)
                ->update(['status' => 'voided']);

            $invoice->update(['status' => 'cancelled']);
        });
    }

    protected function generateInvoiceNumber(): string
    {
        $prefix = 'INV-';
        $year = date('Y');
        $lastInvoice = Invoice::where('invoice_number', 'like', "{$prefix}{$year}%")
            ->orderBy('invoice_number', 'desc')
            ->first();

        if ($lastInvoice) {
            $lastNumber = (int) substr($lastInvoice->invoice_number, -5);
            return $prefix . $year . str_pad($lastNumber + 1, 5, '0', STR_PAD_LEFT);
        }

        return $prefix . $year . '00001';
    }

    protected function generatePaymentNumber(): string
    {
        $prefix = 'PMT-';
        $count = Payment::count() + 1;
        return $prefix . str_pad($count, 6, '0', STR_PAD_LEFT);
    }
}
