<?php

namespace App\Services;

use App\Models\Campaign;
use App\Models\CampaignRecipient;
use App\Models\Contact;
use App\Models\Email;
use App\Models\EmailAccount;
use App\Models\Invoice;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class EmailService
{
    public function send(int $tenantId, array $data, int $userId): Email
    {
        $email = Email::create([
            'tenant_id' => $tenantId,
            'email_account_id' => $data['email_account_id'] ?? null,
            'contact_id' => $data['contact_id'] ?? null,
            'direction' => 'outbound',
            'from_email' => $data['from_email'] ?? config('mail.from.address'),
            'from_name' => $data['from_name'] ?? config('mail.from.name'),
            'to' => $data['to'],
            'cc' => $data['cc'] ?? null,
            'bcc' => $data['bcc'] ?? null,
            'subject' => $data['subject'],
            'body_html' => $data['body'],
            'body_text' => strip_tags($data['body']),
            'tracking_id' => Str::uuid(),
            'created_by' => $userId,
        ]);

        $this->dispatch($email);

        return $email;
    }

    protected function dispatch(Email $email): void
    {
        try {
            Mail::html($email->body_html, function ($message) use ($email) {
                $message->to($email->to)
                    ->subject($email->subject)
                    ->from($email->from_email, $email->from_name);

                if ($email->cc) {
                    $message->cc($email->cc);
                }
                if ($email->bcc) {
                    $message->bcc($email->bcc);
                }
            });

            $email->markAsSent();
        } catch (\Exception $e) {
            $email->markAsFailed($e->getMessage());
        }
    }

    public function sendInvoice(Invoice $invoice, ?string $email = null, ?string $message = null): void
    {
        $recipientEmail = $email ?? $invoice->contact->email;
        $contact = $invoice->contact;

        $body = $message ?? "Dear {$contact->first_name},\n\nPlease find attached invoice #{$invoice->invoice_number}.\n\nAmount Due: {$invoice->currency} " . number_format($invoice->getAmountDue(), 3) . "\nDue Date: {$invoice->due_date->format('M d, Y')}\n\nThank you for your business.";

        $this->send($invoice->tenant_id, [
            'to' => [$recipientEmail],
            'subject' => "Invoice #{$invoice->invoice_number} from " . $invoice->tenant->name,
            'body' => nl2br($body),
            'contact_id' => $contact->id,
        ], $invoice->created_by);
    }

    public function sendCampaignEmail(Campaign $campaign, CampaignRecipient $recipient): void
    {
        $contact = $recipient->contact;
        $content = $this->replaceVariables($campaign->content, $contact);
        $subject = $this->replaceVariables($campaign->subject, $contact);

        Mail::html($content, function ($message) use ($campaign, $recipient, $subject) {
            $message->to($recipient->email)
                ->subject($subject)
                ->from($campaign->from_email, $campaign->from_name);
        });
    }

    protected function replaceVariables(string $content, Contact $contact): string
    {
        $variables = [
            '{{first_name}}' => $contact->first_name,
            '{{last_name}}' => $contact->last_name,
            '{{email}}' => $contact->email,
            '{{company}}' => $contact->company_name,
            '{{full_name}}' => $contact->full_name,
        ];

        return str_replace(array_keys($variables), array_values($variables), $content);
    }

    public function trackOpen(string $trackingId): void
    {
        $email = Email::where('tracking_id', $trackingId)->first();
        if ($email && !$email->opened_at) {
            $email->update([
                'status' => Email::STATUS_OPENED,
                'opened_at' => now(),
                'open_count' => $email->open_count + 1,
            ]);
        }
    }

    public function trackClick(string $trackingId, string $url): void
    {
        $email = Email::where('tracking_id', $trackingId)->first();
        if ($email) {
            $email->update([
                'status' => Email::STATUS_CLICKED,
                'clicked_at' => $email->clicked_at ?? now(),
                'click_count' => $email->click_count + 1,
            ]);
        }
    }
}
