<?php

namespace App\Services;

use App\Models\Bill;
use App\Models\BillItem;
use App\Models\BillPayment;
use Illuminate\Support\Facades\DB;

class BillService
{
    public function create(int $tenantId, array $data, int $userId): Bill
    {
        return DB::transaction(function () use ($tenantId, $data, $userId) {
            $bill = Bill::create([
                'tenant_id' => $tenantId,
                'vendor_id' => $data['vendor_id'],
                'vendor_invoice_number' => $data['vendor_invoice_number'] ?? null,
                'bill_date' => $data['bill_date'],
                'due_date' => $data['due_date'],
                'currency' => $data['currency'] ?? 'BHD',
                'notes' => $data['notes'] ?? null,
                'terms' => $data['terms'] ?? null,
                'discount_type' => $data['discount_type'] ?? null,
                'discount_value' => $data['discount_value'] ?? 0,
                'created_by' => $userId,
            ]);

            $this->syncItems($bill, $data['items']);
            $bill->calculateTotals();
            $bill->save();

            return $bill;
        });
    }

    public function update(Bill $bill, array $data): Bill
    {
        return DB::transaction(function () use ($bill, $data) {
            $bill->update(array_filter([
                'vendor_id' => $data['vendor_id'] ?? null,
                'vendor_invoice_number' => $data['vendor_invoice_number'] ?? null,
                'bill_date' => $data['bill_date'] ?? null,
                'due_date' => $data['due_date'] ?? null,
                'currency' => $data['currency'] ?? null,
                'notes' => $data['notes'] ?? null,
                'terms' => $data['terms'] ?? null,
                'discount_type' => $data['discount_type'] ?? null,
                'discount_value' => $data['discount_value'] ?? null,
            ], fn($v) => !is_null($v)));

            if (isset($data['items'])) {
                $this->syncItems($bill, $data['items']);
            }

            $bill->calculateTotals();
            $bill->save();

            return $bill->fresh();
        });
    }

    protected function syncItems(Bill $bill, array $items): void
    {
        $existingIds = [];

        foreach ($items as $index => $item) {
            if (isset($item['id'])) {
                $billItem = BillItem::find($item['id']);
                $billItem->update(array_merge($item, ['position' => $index + 1]));
                $existingIds[] = $item['id'];
            } else {
                $newItem = $bill->items()->create(array_merge($item, [
                    'tenant_id' => $bill->tenant_id,
                    'position' => $index + 1,
                ]));
                $existingIds[] = $newItem->id;
            }
        }

        $bill->items()->whereNotIn('id', $existingIds)->delete();
    }

    public function recordPayment(Bill $bill, array $data, int $userId): BillPayment
    {
        $payment = BillPayment::create([
            'tenant_id' => $bill->tenant_id,
            'bill_id' => $bill->id,
            'vendor_id' => $bill->vendor_id,
            'bank_account_id' => $data['bank_account_id'] ?? null,
            'payment_date' => $data['payment_date'],
            'amount' => $data['amount'],
            'currency' => $bill->currency,
            'payment_method' => $data['payment_method'],
            'reference' => $data['reference'] ?? null,
            'description' => $data['description'] ?? null,
            'status' => BillPayment::STATUS_COMPLETED,
            'created_by' => $userId,
        ]);

        $bill->updatePaymentStatus();

        return $payment;
    }

    public function duplicate(Bill $bill, int $userId): Bill
    {
        return DB::transaction(function () use ($bill, $userId) {
            $newBill = $bill->replicate([
                'bill_number', 'status', 'amount_paid', 'received_date'
            ]);
            $newBill->status = Bill::STATUS_DRAFT;
            $newBill->amount_paid = 0;
            $newBill->bill_date = now();
            $newBill->due_date = now()->addDays(30);
            $newBill->created_by = $userId;
            $newBill->save();

            foreach ($bill->items as $item) {
                $newItem = $item->replicate();
                $newItem->bill_id = $newBill->id;
                $newItem->save();
            }

            return $newBill;
        });
    }
}
