<?php

namespace App\Notifications;

use App\Models\LeaveRequest;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class LeaveRequestNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public LeaveRequest $leaveRequest,
        public string $action
    ) {}

    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $employee = $this->leaveRequest->employee;
        $leaveType = $this->leaveRequest->leaveType;
        
        $mail = (new MailMessage);

        if ($this->action === 'submitted') {
            $mail->subject("Leave Request: {$employee->full_name}")
                ->greeting("New Leave Request")
                ->line("{$employee->full_name} has submitted a leave request:")
                ->line("Type: {$leaveType->name}")
                ->line("From: {$this->leaveRequest->start_date->format('M d, Y')}")
                ->line("To: {$this->leaveRequest->end_date->format('M d, Y')}")
                ->line("Days: {$this->leaveRequest->days}")
                ->action('Review Request', url("/leave-requests/{$this->leaveRequest->id}"));
        } elseif ($this->action === 'approved') {
            $mail->subject("Leave Request Approved")
                ->greeting("Good news!")
                ->line("Your leave request has been approved.")
                ->line("Type: {$leaveType->name}")
                ->line("From: {$this->leaveRequest->start_date->format('M d, Y')}")
                ->line("To: {$this->leaveRequest->end_date->format('M d, Y')}")
                ->line("Days: {$this->leaveRequest->days}");
        } elseif ($this->action === 'rejected') {
            $mail->subject("Leave Request Declined")
                ->greeting("Leave Request Update")
                ->line("Unfortunately, your leave request has been declined.")
                ->line("Type: {$leaveType->name}")
                ->line("From: {$this->leaveRequest->start_date->format('M d, Y')}")
                ->line("To: {$this->leaveRequest->end_date->format('M d, Y')}");
            
            if ($this->leaveRequest->rejection_reason) {
                $mail->line("Reason: {$this->leaveRequest->rejection_reason}");
            }
        }

        return $mail;
    }

    public function toArray(object $notifiable): array
    {
        $employee = $this->leaveRequest->employee;
        
        return [
            'type' => "leave_request_{$this->action}",
            'leave_request_id' => $this->leaveRequest->id,
            'employee_id' => $employee->id,
            'employee_name' => $employee->full_name,
            'leave_type' => $this->leaveRequest->leaveType->name,
            'start_date' => $this->leaveRequest->start_date->toDateString(),
            'end_date' => $this->leaveRequest->end_date->toDateString(),
            'days' => $this->leaveRequest->days,
            'status' => $this->leaveRequest->status,
            'message' => $this->getMessage(),
        ];
    }

    protected function getMessage(): string
    {
        $employee = $this->leaveRequest->employee;
        
        return match ($this->action) {
            'submitted' => "{$employee->full_name} submitted a leave request",
            'approved' => "Your leave request has been approved",
            'rejected' => "Your leave request has been declined",
            default => "Leave request updated",
        };
    }
}
