<?php

namespace App\Notifications;

use App\Models\Invoice;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class InvoicePaidNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Invoice $invoice
    ) {}

    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject("Invoice {$this->invoice->invoice_number} has been paid")
            ->greeting("Payment Received!")
            ->line("Invoice {$this->invoice->invoice_number} has been paid in full.")
            ->line("Amount: {$this->invoice->currency} " . number_format($this->invoice->total, 3))
            ->line("Customer: {$this->invoice->contact?->name}")
            ->action('View Invoice', url("/invoices/{$this->invoice->id}"))
            ->line('Thank you for your business!');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'invoice_paid',
            'invoice_id' => $this->invoice->id,
            'invoice_number' => $this->invoice->invoice_number,
            'contact_name' => $this->invoice->contact?->name,
            'amount' => $this->invoice->total,
            'currency' => $this->invoice->currency,
            'message' => "Invoice {$this->invoice->invoice_number} has been paid",
        ];
    }
}
