<?php

namespace App\Notifications;

use App\Models\Invoice;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class InvoiceOverdueNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(public Invoice $invoice) {}

    public function via($notifiable): array
    {
        return ['mail', 'database'];
    }

    public function toMail($notifiable): MailMessage
    {
        $daysOverdue = now()->diffInDays($this->invoice->due_date);

        return (new MailMessage)
            ->subject("Payment Reminder: Invoice #{$this->invoice->invoice_number} is Overdue")
            ->greeting("Hello {$notifiable->first_name},")
            ->line("This is a friendly reminder that invoice #{$this->invoice->invoice_number} is now {$daysOverdue} days overdue.")
            ->line("Amount Due: {$this->invoice->currency} " . number_format($this->invoice->getAmountDue(), 3))
            ->line("Original Due Date: " . $this->invoice->due_date->format('M d, Y'))
            ->action('View Invoice', url("/invoices/{$this->invoice->id}"))
            ->line('Please arrange payment at your earliest convenience.');
    }

    public function toArray($notifiable): array
    {
        return [
            'type' => 'invoice_overdue',
            'invoice_id' => $this->invoice->id,
            'invoice_number' => $this->invoice->invoice_number,
            'amount_due' => $this->invoice->getAmountDue(),
            'days_overdue' => now()->diffInDays($this->invoice->due_date),
        ];
    }
}
