<?php

namespace App\Notifications;

use App\Models\Bill;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class BillDueNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(protected Bill $bill) {}

    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $dueDate = $this->bill->due_date->format('M d, Y');
        $amount = number_format($this->bill->total - $this->bill->amount_paid, 3) . ' BHD';
        
        return (new MailMessage)
            ->subject("Bill Due: {$this->bill->bill_number}")
            ->line("Bill {$this->bill->bill_number} from {$this->bill->vendor->name} is due on {$dueDate}.")
            ->line("Amount due: {$amount}")
            ->action('View Bill', url("/bills/{$this->bill->id}"))
            ->line('Please ensure timely payment to avoid late fees.');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'bill_id' => $this->bill->id,
            'bill_number' => $this->bill->bill_number,
            'vendor_name' => $this->bill->vendor->name,
            'amount_due' => $this->bill->total - $this->bill->amount_paid,
            'due_date' => $this->bill->due_date->toDateString(),
        ];
    }
}
