<?php

namespace App\Models;

use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Traits\BelongsToTenant;

class Workflow extends BaseModel
{
    use SoftDeletes, BelongsToTenant;

    protected $casts = [
        'trigger_config' => 'array',
        'steps' => 'array',
        'is_active' => 'boolean',
        'last_triggered_at' => 'datetime',
    ];

    public function enrollments(): HasMany
    {
        return $this->hasMany(WorkflowEnrollment::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function enroll(Contact $contact): WorkflowEnrollment
    {
        return $this->enrollments()->create([
            'contact_id' => $contact->id,
            'current_step' => 0,
            'status' => 'active',
        ]);
    }
}

class WorkflowEnrollment extends BaseModel
{
    protected $casts = [
        'next_action_at' => 'datetime',
        'step_data' => 'array',
    ];

    public function workflow(): BelongsTo
    {
        return $this->belongsTo(Workflow::class);
    }

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }
}

class Campaign extends BaseModel
{
    use SoftDeletes, BelongsToTenant;

    protected $casts = [
        'audience_filters' => 'array',
        'scheduled_at' => 'datetime',
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
    ];

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function recipients(): HasMany
    {
        return $this->hasMany(CampaignRecipient::class);
    }

    public function getOpenRate(): float
    {
        return $this->sent_count > 0 ? ($this->opened_count / $this->sent_count) * 100 : 0;
    }
}

class Bill extends BaseModel
{
    use SoftDeletes, BelongsToTenant;

    protected $casts = [
        'bill_date' => 'date',
        'due_date' => 'date',
        'subtotal' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'total' => 'decimal:2',
        'amount_paid' => 'decimal:2',
    ];

    public function vendor(): BelongsTo
    {
        return $this->belongsTo(Vendor::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(BillItem::class);
    }

    public function payments(): HasMany
    {
        return $this->hasMany(BillPayment::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function getAmountDue(): float
    {
        return $this->total - $this->amount_paid;
    }

    public function isPaid(): bool
    {
        return $this->getAmountDue() <= 0;
    }

    public function isOverdue(): bool
    {
        return !$this->isPaid() && $this->due_date->isPast();
    }
}

class Payment extends BaseModel
{
    use SoftDeletes, BelongsToTenant;

    protected $casts = [
        'payment_date' => 'date',
        'amount' => 'decimal:2',
        'gateway_response' => 'array',
    ];

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function invoice(): BelongsTo
    {
        return $this->belongsTo(Invoice::class);
    }

    public function bankAccount(): BelongsTo
    {
        return $this->belongsTo(ChartOfAccount::class, 'bank_account_id');
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }
}
