<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class WhatsappMessage extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'contact_id',
        'campaign_id',
        'direction',
        'from_number',
        'to_number',
        'message_type',
        'content',
        'media_url',
        'media_mime_type',
        'media_filename',
        'template_name',
        'template_params',
        'status',
        'provider_id',
        'provider_response',
        'sent_at',
        'delivered_at',
        'read_at',
        'failed_at',
        'error_message',
    ];

    protected $casts = [
        'template_params' => 'array',
        'provider_response' => 'array',
        'sent_at' => 'datetime',
        'delivered_at' => 'datetime',
        'read_at' => 'datetime',
        'failed_at' => 'datetime',
    ];

    const DIRECTION_INBOUND = 'inbound';
    const DIRECTION_OUTBOUND = 'outbound';

    const TYPE_TEXT = 'text';
    const TYPE_IMAGE = 'image';
    const TYPE_DOCUMENT = 'document';
    const TYPE_AUDIO = 'audio';
    const TYPE_VIDEO = 'video';
    const TYPE_TEMPLATE = 'template';
    const TYPE_INTERACTIVE = 'interactive';
    const TYPE_LOCATION = 'location';
    const TYPE_CONTACTS = 'contacts';

    const STATUS_QUEUED = 'queued';
    const STATUS_SENT = 'sent';
    const STATUS_DELIVERED = 'delivered';
    const STATUS_READ = 'read';
    const STATUS_FAILED = 'failed';
    const STATUS_RECEIVED = 'received';

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function campaign(): BelongsTo
    {
        return $this->belongsTo(Campaign::class);
    }

    public function markAsSent(string $providerId = null): void
    {
        $this->update([
            'status' => self::STATUS_SENT,
            'sent_at' => now(),
            'provider_id' => $providerId,
        ]);
    }

    public function markAsDelivered(): void
    {
        $this->update([
            'status' => self::STATUS_DELIVERED,
            'delivered_at' => now(),
        ]);
    }

    public function markAsRead(): void
    {
        $this->update([
            'status' => self::STATUS_READ,
            'read_at' => now(),
        ]);
    }

    public function markAsFailed(string $reason = null): void
    {
        $this->update([
            'status' => self::STATUS_FAILED,
            'failed_at' => now(),
            'error_message' => $reason,
        ]);
    }

    public function hasMedia(): bool
    {
        return in_array($this->message_type, [
            self::TYPE_IMAGE,
            self::TYPE_DOCUMENT,
            self::TYPE_AUDIO,
            self::TYPE_VIDEO,
        ]);
    }

    public function isTemplate(): bool
    {
        return $this->message_type === self::TYPE_TEMPLATE;
    }

    public function scopeOutbound($query)
    {
        return $query->where('direction', self::DIRECTION_OUTBOUND);
    }

    public function scopeInbound($query)
    {
        return $query->where('direction', self::DIRECTION_INBOUND);
    }

    public function scopeConversation($query, string $phoneNumber)
    {
        return $query->where(function ($q) use ($phoneNumber) {
            $q->where('from_number', $phoneNumber)
              ->orWhere('to_number', $phoneNumber);
        })->orderBy('created_at');
    }
}
