<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;

class TaxRate extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'name',
        'rate',
        'type',
        'is_compound',
        'is_inclusive',
        'description',
        'is_default',
        'is_active',
    ];

    protected $casts = [
        'rate' => 'decimal:4',
        'is_compound' => 'boolean',
        'is_inclusive' => 'boolean',
        'is_default' => 'boolean',
        'is_active' => 'boolean',
    ];

    const TYPE_SALES = 'sales';
    const TYPE_PURCHASE = 'purchase';
    const TYPE_BOTH = 'both';

    public static function getDefault($tenantId, $type = self::TYPE_SALES)
    {
        return static::where('tenant_id', $tenantId)
            ->where('is_default', true)
            ->where('is_active', true)
            ->where(function ($query) use ($type) {
                $query->where('type', $type)->orWhere('type', self::TYPE_BOTH);
            })
            ->first();
    }

    public function calculateTax(float $amount): float
    {
        if ($this->is_inclusive) {
            return $amount - ($amount / (1 + ($this->rate / 100)));
        }
        
        return $amount * ($this->rate / 100);
    }

    public function getAmountWithTax(float $amount): float
    {
        if ($this->is_inclusive) {
            return $amount;
        }
        
        return $amount + $this->calculateTax($amount);
    }

    public function getAmountWithoutTax(float $amount): float
    {
        if ($this->is_inclusive) {
            return $amount / (1 + ($this->rate / 100));
        }
        
        return $amount;
    }
}
