<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class TaskActivity extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'task_id',
        'user_id',
        'action',
        'field',
        'old_value',
        'new_value',
        'metadata',
    ];

    protected $casts = [
        'metadata' => 'array',
    ];

    const ACTION_CREATED = 'created';
    const ACTION_UPDATED = 'updated';
    const ACTION_DELETED = 'deleted';
    const ACTION_COMPLETED = 'completed';
    const ACTION_REOPENED = 'reopened';
    const ACTION_ASSIGNED = 'assigned';
    const ACTION_UNASSIGNED = 'unassigned';
    const ACTION_MOVED = 'moved';
    const ACTION_COMMENTED = 'commented';
    const ACTION_ATTACHMENT_ADDED = 'attachment_added';
    const ACTION_ATTACHMENT_REMOVED = 'attachment_removed';
    const ACTION_DUE_DATE_CHANGED = 'due_date_changed';
    const ACTION_PRIORITY_CHANGED = 'priority_changed';
    const ACTION_STATUS_CHANGED = 'status_changed';

    protected static function booted(): void
    {
        static::creating(function ($activity) {
            if (empty($activity->user_id)) {
                $activity->user_id = auth()->id();
            }
        });
    }

    public function task(): BelongsTo
    {
        return $this->belongsTo(Task::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function getDescription(): string
    {
        $userName = $this->user?->name ?? 'Someone';

        return match ($this->action) {
            self::ACTION_CREATED => "{$userName} created this task",
            self::ACTION_UPDATED => "{$userName} updated {$this->field}",
            self::ACTION_DELETED => "{$userName} deleted this task",
            self::ACTION_COMPLETED => "{$userName} completed this task",
            self::ACTION_REOPENED => "{$userName} reopened this task",
            self::ACTION_ASSIGNED => "{$userName} assigned to {$this->new_value}",
            self::ACTION_UNASSIGNED => "{$userName} unassigned {$this->old_value}",
            self::ACTION_MOVED => "{$userName} moved from {$this->old_value} to {$this->new_value}",
            self::ACTION_COMMENTED => "{$userName} commented",
            self::ACTION_ATTACHMENT_ADDED => "{$userName} added attachment",
            self::ACTION_ATTACHMENT_REMOVED => "{$userName} removed attachment",
            self::ACTION_DUE_DATE_CHANGED => "{$userName} changed due date from {$this->old_value} to {$this->new_value}",
            self::ACTION_PRIORITY_CHANGED => "{$userName} changed priority from {$this->old_value} to {$this->new_value}",
            self::ACTION_STATUS_CHANGED => "{$userName} changed status from {$this->old_value} to {$this->new_value}",
            default => "{$userName} performed {$this->action}",
        };
    }

    public static function log(Task $task, string $action, array $data = []): self
    {
        return static::create([
            'tenant_id' => $task->tenant_id,
            'task_id' => $task->id,
            'action' => $action,
            'field' => $data['field'] ?? null,
            'old_value' => $data['old_value'] ?? null,
            'new_value' => $data['new_value'] ?? null,
            'metadata' => $data['metadata'] ?? null,
        ]);
    }
}
