<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use App\Traits\BelongsToTenant;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class Project extends Model
{
    use HasFactory, SoftDeletes, BelongsToTenant, LogsActivity;

    protected $fillable = [
        'tenant_id',
        'team_id',
        'contact_id',
        'opportunity_id',
        'owner_id',
        'status_id',
        'name',
        'code',
        'description',
        'color',
        'start_date',
        'due_date',
        'completed_date',
        'budget',
        'hourly_rate',
        'currency',
        'visibility',
        'billing_type',
        'progress',
        'tasks_count',
        'completed_tasks_count',
        'settings',
        'custom_fields',
        'is_template',
        'template_id',
    ];

    protected $casts = [
        'start_date' => 'date',
        'due_date' => 'date',
        'completed_date' => 'date',
        'budget' => 'decimal:2',
        'hourly_rate' => 'decimal:2',
        'progress' => 'decimal:2',
        'settings' => 'array',
        'custom_fields' => 'array',
        'is_template' => 'boolean',
    ];

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['name', 'status_id', 'owner_id', 'due_date'])
            ->logOnlyDirty();
    }

    // Relationships
    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class);
    }

    public function team(): BelongsTo
    {
        return $this->belongsTo(Team::class);
    }

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function opportunity(): BelongsTo
    {
        return $this->belongsTo(Opportunity::class);
    }

    public function owner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    public function status(): BelongsTo
    {
        return $this->belongsTo(ProjectStatus::class, 'status_id');
    }

    public function template(): BelongsTo
    {
        return $this->belongsTo(Project::class, 'template_id');
    }

    public function members(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'project_members')
            ->withPivot('role', 'hourly_rate')
            ->withTimestamps();
    }

    public function taskStatuses(): HasMany
    {
        return $this->hasMany(TaskStatus::class)->orderBy('position');
    }

    public function taskLists(): HasMany
    {
        return $this->hasMany(TaskList::class)->orderBy('position');
    }

    public function tasks(): HasMany
    {
        return $this->hasMany(Task::class);
    }

    public function milestones(): HasMany
    {
        return $this->hasMany(Milestone::class);
    }

    public function timeEntries(): HasMany
    {
        return $this->hasMany(TimeEntry::class);
    }

    public function goals(): BelongsToMany
    {
        return $this->belongsToMany(Goal::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->whereNull('completed_date');
    }

    public function scopeCompleted($query)
    {
        return $query->whereNotNull('completed_date');
    }

    public function scopeOverdue($query)
    {
        return $query->whereNull('completed_date')
            ->whereNotNull('due_date')
            ->where('due_date', '<', now());
    }

    public function scopeVisible($query, User $user)
    {
        return $query->where(function ($q) use ($user) {
            $q->where('visibility', 'public')
                ->orWhere('owner_id', $user->id)
                ->orWhereHas('members', fn($m) => $m->where('user_id', $user->id));
        });
    }

    // Helpers
    public function isOverdue(): bool
    {
        return $this->due_date && $this->due_date->isPast() && !$this->completed_date;
    }

    public function isCompleted(): bool
    {
        return $this->completed_date !== null;
    }

    public function updateProgress(): void
    {
        $totalTasks = $this->tasks()->count();
        $completedTasks = $this->tasks()->whereNotNull('completed_at')->count();
        
        $progress = $totalTasks > 0 ? ($completedTasks / $totalTasks) * 100 : 0;
        
        $this->update([
            'progress' => round($progress, 2),
            'tasks_count' => $totalTasks,
            'completed_tasks_count' => $completedTasks,
        ]);
    }

    public function markAsCompleted(): void
    {
        $this->update(['completed_date' => now()]);
    }

    public function getTotalLoggedHours(): float
    {
        return $this->timeEntries()->sum('hours');
    }

    public function getTotalBillableAmount(): float
    {
        return $this->timeEntries()
            ->where('is_billable', true)
            ->sum('amount');
    }

    public function getBudgetUsedPercentage(): float
    {
        if (!$this->budget || $this->budget == 0) {
            return 0;
        }

        $spent = $this->getTotalBillableAmount();
        return min(100, ($spent / $this->budget) * 100);
    }

    public function hasMember(User $user): bool
    {
        return $this->members()->where('user_id', $user->id)->exists();
    }

    public function addMember(User $user, string $role = 'member'): void
    {
        if (!$this->hasMember($user)) {
            $this->members()->attach($user->id, [
                'role' => $role,
                'hourly_rate' => $user->hourly_rate,
            ]);
        }
    }

    public function removeMember(User $user): void
    {
        $this->members()->detach($user->id);
    }
}
