<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Traits\BelongsToTenant;

class PayPeriod extends BaseModel
{
    use BelongsToTenant;

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'pay_date' => 'date',
    ];

    public function payrollRuns(): HasMany
    {
        return $this->hasMany(PayrollRun::class);
    }
}

class PayrollRun extends BaseModel
{
    use BelongsToTenant;

    protected $casts = [
        'total_gross' => 'decimal:2',
        'total_deductions' => 'decimal:2',
        'total_net' => 'decimal:2',
        'total_employer_cost' => 'decimal:2',
        'approved_at' => 'datetime',
        'paid_at' => 'datetime',
    ];

    public function payPeriod(): BelongsTo
    {
        return $this->belongsTo(PayPeriod::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(PayrollItem::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function approvedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function calculate(): void
    {
        $employees = Employee::where('status', 'active')->get();
        
        foreach ($employees as $employee) {
            $this->calculateForEmployee($employee);
        }

        $this->recalculateTotals();
    }

    protected function calculateForEmployee(Employee $employee): PayrollItem
    {
        $item = $this->items()->updateOrCreate(
            ['employee_id' => $employee->id],
            [
                'basic_salary' => $employee->salary_amount,
                'gross_salary' => $employee->salary_amount,
                'net_salary' => $employee->salary_amount,
            ]
        );

        return $item;
    }

    public function recalculateTotals(): void
    {
        $this->update([
            'total_gross' => $this->items()->sum('gross_salary'),
            'total_deductions' => $this->items()->sum('total_deductions'),
            'total_net' => $this->items()->sum('net_salary'),
            'total_employer_cost' => $this->items()->sum('employer_cost'),
            'employee_count' => $this->items()->count(),
        ]);
    }
}

class PayrollItem extends BaseModel
{
    protected $casts = [
        'basic_salary' => 'decimal:2',
        'housing_allowance' => 'decimal:2',
        'transport_allowance' => 'decimal:2',
        'other_allowances' => 'decimal:2',
        'overtime_hours' => 'decimal:2',
        'overtime_amount' => 'decimal:2',
        'bonus' => 'decimal:2',
        'commission' => 'decimal:2',
        'gross_salary' => 'decimal:2',
        'social_insurance_employee' => 'decimal:2',
        'social_insurance_employer' => 'decimal:2',
        'income_tax' => 'decimal:2',
        'loan_deduction' => 'decimal:2',
        'other_deductions' => 'decimal:2',
        'total_deductions' => 'decimal:2',
        'net_salary' => 'decimal:2',
        'employer_cost' => 'decimal:2',
        'breakdown' => 'array',
    ];

    public function payrollRun(): BelongsTo
    {
        return $this->belongsTo(PayrollRun::class);
    }

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }
}

class LeaveType extends BaseModel
{
    use BelongsToTenant;

    protected $casts = [
        'is_paid' => 'boolean',
        'is_carry_forward' => 'boolean',
        'requires_approval' => 'boolean',
        'requires_document' => 'boolean',
        'is_active' => 'boolean',
    ];

    public function leaveRequests(): HasMany
    {
        return $this->hasMany(LeaveRequest::class);
    }

    public function leaveBalances(): HasMany
    {
        return $this->hasMany(LeaveBalance::class);
    }
}

class LeaveRequest extends BaseModel
{
    use BelongsToTenant;

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'days' => 'decimal:2',
        'approved_at' => 'datetime',
    ];

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }

    public function leaveType(): BelongsTo
    {
        return $this->belongsTo(LeaveType::class);
    }

    public function approvedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function approve(User $approver): bool
    {
        if ($this->status !== 'pending') {
            return false;
        }

        $this->update([
            'status' => 'approved',
            'approved_by' => $approver->id,
            'approved_at' => now(),
        ]);

        // Update leave balance
        LeaveBalance::where('employee_id', $this->employee_id)
            ->where('leave_type_id', $this->leave_type_id)
            ->where('year', $this->start_date->year)
            ->decrement('balance', $this->days);

        return true;
    }

    public function reject(User $approver, string $reason): bool
    {
        return $this->update([
            'status' => 'rejected',
            'approved_by' => $approver->id,
            'approved_at' => now(),
            'rejection_reason' => $reason,
        ]);
    }
}

class LeaveBalance extends BaseModel
{
    protected $casts = [
        'entitled' => 'decimal:2',
        'carried_forward' => 'decimal:2',
        'used' => 'decimal:2',
        'pending' => 'decimal:2',
        'balance' => 'decimal:2',
    ];

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }

    public function leaveType(): BelongsTo
    {
        return $this->belongsTo(LeaveType::class);
    }
}

class Attendance extends BaseModel
{
    use BelongsToTenant;

    protected $table = 'attendance';

    protected $casts = [
        'date' => 'date',
        'check_in' => 'datetime:H:i',
        'check_out' => 'datetime:H:i',
        'work_hours' => 'decimal:2',
        'overtime_hours' => 'decimal:2',
        'break_hours' => 'decimal:2',
        'is_manual' => 'boolean',
    ];

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }

    public function calculateHours(): void
    {
        if ($this->check_in && $this->check_out) {
            $diff = $this->check_in->diffInMinutes($this->check_out);
            $this->work_hours = round(($diff - ($this->break_hours * 60)) / 60, 2);
            
            if ($this->work_hours > 8) {
                $this->overtime_hours = $this->work_hours - 8;
            }
        }
    }
}

class EmployeeLoan extends BaseModel
{
    use BelongsToTenant;

    protected $casts = [
        'amount' => 'decimal:2',
        'paid_amount' => 'decimal:2',
        'remaining_amount' => 'decimal:2',
        'monthly_deduction' => 'decimal:2',
        'start_date' => 'date',
        'end_date' => 'date',
    ];

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }

    public function approvedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function recordPayment(float $amount): void
    {
        $this->paid_amount += $amount;
        $this->remaining_amount = $this->amount - $this->paid_amount;
        $this->paid_installments++;

        if ($this->remaining_amount <= 0) {
            $this->status = 'completed';
        }

        $this->save();
    }
}
