<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Milestone extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'project_id',
        'name',
        'description',
        'due_date',
        'completed_at',
        'status',
        'color',
        'position',
    ];

    protected $casts = [
        'due_date' => 'date',
        'completed_at' => 'datetime',
        'position' => 'integer',
    ];

    const STATUS_PENDING = 'pending';
    const STATUS_IN_PROGRESS = 'in_progress';
    const STATUS_COMPLETED = 'completed';

    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    public function tasks(): HasMany
    {
        return $this->hasMany(Task::class);
    }

    public function getProgress(): float
    {
        $total = $this->tasks()->count();
        
        if ($total === 0) {
            return 0;
        }

        $completed = $this->tasks()
            ->whereHas('status', fn($q) => $q->where('is_completed', true))
            ->count();

        return ($completed / $total) * 100;
    }

    public function updateStatus(): void
    {
        $progress = $this->getProgress();

        if ($progress >= 100) {
            $this->update([
                'status' => self::STATUS_COMPLETED,
                'completed_at' => now(),
            ]);
        } elseif ($progress > 0) {
            $this->update(['status' => self::STATUS_IN_PROGRESS]);
        } else {
            $this->update(['status' => self::STATUS_PENDING]);
        }
    }

    public function isOverdue(): bool
    {
        return $this->due_date && 
               $this->due_date->isPast() && 
               $this->status !== self::STATUS_COMPLETED;
    }

    public function complete(): void
    {
        $this->update([
            'status' => self::STATUS_COMPLETED,
            'completed_at' => now(),
        ]);
    }

    public function reopen(): void
    {
        $this->update([
            'status' => self::STATUS_IN_PROGRESS,
            'completed_at' => null,
        ]);
    }
}
