<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;

class Holiday extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'name',
        'date',
        'type',
        'is_recurring',
        'applies_to_all',
        'department_ids',
    ];

    protected $casts = [
        'date' => 'date',
        'is_recurring' => 'boolean',
        'applies_to_all' => 'boolean',
        'department_ids' => 'array',
    ];

    const TYPE_PUBLIC = 'public';
    const TYPE_COMPANY = 'company';
    const TYPE_OPTIONAL = 'optional';

    public function appliesToEmployee(Employee $employee): bool
    {
        if ($this->applies_to_all) {
            return true;
        }

        if (empty($this->department_ids)) {
            return true;
        }

        return in_array($employee->department_id, $this->department_ids);
    }

    public static function isHoliday($tenantId, $date): bool
    {
        $date = \Carbon\Carbon::parse($date);
        
        return static::where('tenant_id', $tenantId)
            ->where(function ($query) use ($date) {
                $query->where('date', $date->format('Y-m-d'))
                    ->orWhere(function ($q) use ($date) {
                        $q->where('is_recurring', true)
                            ->whereMonth('date', $date->month)
                            ->whereDay('date', $date->day);
                    });
            })
            ->exists();
    }

    public static function getForYear($tenantId, int $year)
    {
        return static::where('tenant_id', $tenantId)
            ->where(function ($query) use ($year) {
                $query->whereYear('date', $year)
                    ->orWhere('is_recurring', true);
            })
            ->orderBy('date')
            ->get()
            ->map(function ($holiday) use ($year) {
                if ($holiday->is_recurring) {
                    $holiday->date = $holiday->date->setYear($year);
                }
                return $holiday;
            })
            ->sortBy('date')
            ->values();
    }

    public static function seedBahrainHolidays($tenantId): void
    {
        $holidays = [
            ['name' => 'New Year\'s Day', 'date' => '2024-01-01', 'is_recurring' => true],
            ['name' => 'Labour Day', 'date' => '2024-05-01', 'is_recurring' => true],
            ['name' => 'Eid Al Fitr', 'date' => '2024-04-10', 'is_recurring' => false],
            ['name' => 'Eid Al Adha', 'date' => '2024-06-16', 'is_recurring' => false],
            ['name' => 'Islamic New Year', 'date' => '2024-07-07', 'is_recurring' => false],
            ['name' => 'Ashura', 'date' => '2024-07-16', 'is_recurring' => false],
            ['name' => 'Prophet\'s Birthday', 'date' => '2024-09-15', 'is_recurring' => false],
            ['name' => 'National Day', 'date' => '2024-12-16', 'is_recurring' => true],
            ['name' => 'National Day', 'date' => '2024-12-17', 'is_recurring' => true],
        ];

        foreach ($holidays as $holiday) {
            static::firstOrCreate(
                ['tenant_id' => $tenantId, 'name' => $holiday['name'], 'date' => $holiday['date']],
                array_merge($holiday, ['tenant_id' => $tenantId, 'type' => self::TYPE_PUBLIC, 'applies_to_all' => true])
            );
        }
    }
}
