<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Goal extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'parent_id',
        'owner_id',
        'name',
        'description',
        'type',
        'period',
        'start_date',
        'end_date',
        'target_value',
        'current_value',
        'unit',
        'progress',
        'status',
        'color',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'target_value' => 'decimal:2',
        'current_value' => 'decimal:2',
        'progress' => 'decimal:2',
    ];

    const TYPE_OBJECTIVE = 'objective';
    const TYPE_KEY_RESULT = 'key_result';
    const TYPE_INITIATIVE = 'initiative';

    const PERIOD_WEEKLY = 'weekly';
    const PERIOD_MONTHLY = 'monthly';
    const PERIOD_QUARTERLY = 'quarterly';
    const PERIOD_YEARLY = 'yearly';

    const STATUS_NOT_STARTED = 'not_started';
    const STATUS_ON_TRACK = 'on_track';
    const STATUS_AT_RISK = 'at_risk';
    const STATUS_BEHIND = 'behind';
    const STATUS_COMPLETED = 'completed';

    public function parent(): BelongsTo
    {
        return $this->belongsTo(Goal::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(Goal::class, 'parent_id');
    }

    public function owner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    public function projects(): BelongsToMany
    {
        return $this->belongsToMany(Project::class, 'goal_project');
    }

    public function calculateProgress(): float
    {
        if ($this->target_value <= 0) {
            return 0;
        }

        return min(100, ($this->current_value / $this->target_value) * 100);
    }

    public function updateProgress(): void
    {
        if ($this->type === self::TYPE_OBJECTIVE) {
            // Calculate from key results
            $keyResults = $this->children()->where('type', self::TYPE_KEY_RESULT)->get();
            
            if ($keyResults->isNotEmpty()) {
                $this->progress = $keyResults->avg('progress');
            }
        } else {
            $this->progress = $this->calculateProgress();
        }

        $this->updateStatus();
        $this->save();

        // Update parent if exists
        if ($this->parent_id) {
            $this->parent->updateProgress();
        }
    }

    public function updateStatus(): void
    {
        if ($this->progress >= 100) {
            $this->status = self::STATUS_COMPLETED;
            return;
        }

        if ($this->progress == 0) {
            $this->status = self::STATUS_NOT_STARTED;
            return;
        }

        // Calculate expected progress based on time elapsed
        $totalDays = $this->start_date->diffInDays($this->end_date);
        $elapsedDays = $this->start_date->diffInDays(now());
        $expectedProgress = $totalDays > 0 ? ($elapsedDays / $totalDays) * 100 : 0;

        if ($this->progress >= $expectedProgress * 0.9) {
            $this->status = self::STATUS_ON_TRACK;
        } elseif ($this->progress >= $expectedProgress * 0.7) {
            $this->status = self::STATUS_AT_RISK;
        } else {
            $this->status = self::STATUS_BEHIND;
        }
    }

    public function scopeObjectives($query)
    {
        return $query->where('type', self::TYPE_OBJECTIVE);
    }

    public function scopeKeyResults($query)
    {
        return $query->where('type', self::TYPE_KEY_RESULT);
    }

    public function scopeForPeriod($query, string $period)
    {
        return $query->where('period', $period);
    }

    public function scopeActive($query)
    {
        return $query->where('end_date', '>=', now())
            ->where('status', '!=', self::STATUS_COMPLETED);
    }
}
