<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class FormSubmission extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'form_id',
        'contact_id',
        'data',
        'ip_address',
        'user_agent',
        'referrer',
        'utm_source',
        'utm_medium',
        'utm_campaign',
        'utm_term',
        'utm_content',
    ];

    protected $casts = [
        'data' => 'array',
    ];

    protected static function booted(): void
    {
        static::created(function ($submission) {
            $submission->form->increment('submission_count');
        });

        static::deleted(function ($submission) {
            $submission->form->decrement('submission_count');
        });
    }

    public function form(): BelongsTo
    {
        return $this->belongsTo(Form::class);
    }

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function getValue(string $fieldName, $default = null)
    {
        return $this->data[$fieldName] ?? $default;
    }

    public function getEmail(): ?string
    {
        // Try common email field names
        $emailFields = ['email', 'email_address', 'e-mail', 'contact_email'];
        
        foreach ($emailFields as $field) {
            if ($email = $this->getValue($field)) {
                return $email;
            }
        }
        
        return null;
    }

    public function getName(): ?string
    {
        // Try to construct full name
        $firstName = $this->getValue('first_name') ?? $this->getValue('firstname') ?? '';
        $lastName = $this->getValue('last_name') ?? $this->getValue('lastname') ?? '';
        
        if ($firstName || $lastName) {
            return trim("{$firstName} {$lastName}");
        }
        
        return $this->getValue('name') ?? $this->getValue('full_name');
    }

    public function getPhone(): ?string
    {
        $phoneFields = ['phone', 'phone_number', 'telephone', 'mobile', 'cell'];
        
        foreach ($phoneFields as $field) {
            if ($phone = $this->getValue($field)) {
                return $phone;
            }
        }
        
        return null;
    }

    public function createOrUpdateContact(): ?Contact
    {
        $email = $this->getEmail();
        
        if (!$email) {
            return null;
        }

        $contact = Contact::firstOrNew([
            'tenant_id' => $this->tenant_id,
            'email' => $email,
        ]);

        $contact->fill([
            'first_name' => $this->getValue('first_name') ?? $contact->first_name,
            'last_name' => $this->getValue('last_name') ?? $contact->last_name,
            'phone' => $this->getPhone() ?? $contact->phone,
            'company_name' => $this->getValue('company') ?? $this->getValue('company_name') ?? $contact->company_name,
            'source' => 'form',
        ]);

        $contact->save();

        $this->update(['contact_id' => $contact->id]);

        return $contact;
    }

    public function hasUtmData(): bool
    {
        return $this->utm_source || $this->utm_medium || $this->utm_campaign;
    }

    public function getUtmData(): array
    {
        return [
            'source' => $this->utm_source,
            'medium' => $this->utm_medium,
            'campaign' => $this->utm_campaign,
            'term' => $this->utm_term,
            'content' => $this->utm_content,
        ];
    }
}
