<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class Expense extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'account_id',
        'vendor_id',
        'contact_id',
        'bill_id',
        'expense_number',
        'expense_date',
        'amount',
        'currency',
        'exchange_rate',
        'tax_rate_id',
        'tax_amount',
        'total',
        'category',
        'description',
        'receipt_path',
        'is_billable',
        'is_billed',
        'is_reimbursable',
        'is_reimbursed',
        'payment_method',
        'reference',
        'status',
        'approved_at',
        'approved_by',
        'rejected_at',
        'rejected_by',
        'rejection_reason',
        'created_by',
    ];

    protected $casts = [
        'expense_date' => 'date',
        'amount' => 'decimal:3',
        'exchange_rate' => 'decimal:6',
        'tax_amount' => 'decimal:3',
        'total' => 'decimal:3',
        'is_billable' => 'boolean',
        'is_billed' => 'boolean',
        'is_reimbursable' => 'boolean',
        'is_reimbursed' => 'boolean',
        'approved_at' => 'datetime',
        'rejected_at' => 'datetime',
    ];

    const STATUS_PENDING = 'pending';
    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';
    const STATUS_PAID = 'paid';

    protected static function booted(): void
    {
        static::creating(function ($expense) {
            if (empty($expense->expense_number)) {
                $expense->expense_number = static::generateNumber($expense->tenant_id);
            }
            if (empty($expense->status)) {
                $expense->status = self::STATUS_PENDING;
            }
        });
    }

    public static function generateNumber($tenantId): string
    {
        $latest = static::where('tenant_id', $tenantId)
            ->whereYear('created_at', now()->year)
            ->latest('id')
            ->first();

        $number = $latest ? (int) substr($latest->expense_number, -5) + 1 : 1;
        return 'EXP-' . now()->format('Y') . '-' . str_pad($number, 5, '0', STR_PAD_LEFT);
    }

    public function account(): BelongsTo
    {
        return $this->belongsTo(ChartOfAccount::class, 'account_id');
    }

    public function vendor(): BelongsTo
    {
        return $this->belongsTo(Vendor::class);
    }

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function bill(): BelongsTo
    {
        return $this->belongsTo(Bill::class);
    }

    public function taxRate(): BelongsTo
    {
        return $this->belongsTo(TaxRate::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function approvedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function rejectedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'rejected_by');
    }

    public function calculateTotal(): void
    {
        if ($this->taxRate) {
            $this->tax_amount = $this->amount * ($this->taxRate->rate / 100);
        } else {
            $this->tax_amount = 0;
        }
        
        $this->total = $this->amount + $this->tax_amount;
    }

    public function approve(User $user): void
    {
        $this->update([
            'status' => self::STATUS_APPROVED,
            'approved_at' => now(),
            'approved_by' => $user->id,
        ]);
    }

    public function reject(User $user, string $reason = null): void
    {
        $this->update([
            'status' => self::STATUS_REJECTED,
            'rejected_at' => now(),
            'rejected_by' => $user->id,
            'rejection_reason' => $reason,
        ]);
    }

    public function markAsPaid(): void
    {
        $this->update(['status' => self::STATUS_PAID]);
    }

    public function markAsBilled(): void
    {
        $this->update(['is_billed' => true]);
    }

    public function markAsReimbursed(): void
    {
        $this->update(['is_reimbursed' => true]);
    }
}
