<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class EmailAccount extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'name',
        'email',
        'provider',
        'credentials',
        'smtp_host',
        'smtp_port',
        'smtp_encryption',
        'imap_host',
        'imap_port',
        'imap_encryption',
        'is_default',
        'is_active',
        'daily_limit',
        'sent_today',
        'last_sent_at',
        'last_synced_at',
    ];

    protected $casts = [
        'credentials' => 'encrypted:array',
        'is_default' => 'boolean',
        'is_active' => 'boolean',
        'daily_limit' => 'integer',
        'sent_today' => 'integer',
        'last_sent_at' => 'datetime',
        'last_synced_at' => 'datetime',
    ];

    protected $hidden = [
        'credentials',
    ];

    const PROVIDER_SMTP = 'smtp';
    const PROVIDER_SENDGRID = 'sendgrid';
    const PROVIDER_MAILGUN = 'mailgun';
    const PROVIDER_SES = 'ses';
    const PROVIDER_GMAIL = 'gmail';
    const PROVIDER_OUTLOOK = 'outlook';

    public function emails(): HasMany
    {
        return $this->hasMany(Email::class);
    }

    public function canSend(): bool
    {
        if (!$this->is_active) {
            return false;
        }

        if ($this->daily_limit && $this->sent_today >= $this->daily_limit) {
            return false;
        }

        return true;
    }

    public function incrementSentCount(): void
    {
        $this->increment('sent_today');
        $this->update(['last_sent_at' => now()]);
    }

    public function resetDailyCount(): void
    {
        $this->update(['sent_today' => 0]);
    }

    public static function getDefault($tenantId): ?self
    {
        return static::where('tenant_id', $tenantId)
            ->where('is_default', true)
            ->where('is_active', true)
            ->first();
    }

    public function getSmtpConfig(): array
    {
        return [
            'host' => $this->smtp_host,
            'port' => $this->smtp_port,
            'encryption' => $this->smtp_encryption,
            'username' => $this->credentials['username'] ?? null,
            'password' => $this->credentials['password'] ?? null,
        ];
    }

    public function getImapConfig(): array
    {
        return [
            'host' => $this->imap_host,
            'port' => $this->imap_port,
            'encryption' => $this->imap_encryption,
            'username' => $this->credentials['username'] ?? null,
            'password' => $this->credentials['password'] ?? null,
        ];
    }
}
