<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class BankAccount extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'account_id',
        'name',
        'bank_name',
        'account_number',
        'routing_number',
        'iban',
        'swift_code',
        'currency',
        'opening_balance',
        'current_balance',
        'is_default',
        'is_active',
        'plaid_access_token',
        'plaid_item_id',
        'plaid_account_id',
        'last_synced_at',
        'description',
    ];

    protected $casts = [
        'opening_balance' => 'decimal:3',
        'current_balance' => 'decimal:3',
        'is_default' => 'boolean',
        'is_active' => 'boolean',
        'last_synced_at' => 'datetime',
    ];

    protected $hidden = [
        'plaid_access_token',
    ];

    public function chartAccount(): BelongsTo
    {
        return $this->belongsTo(ChartOfAccount::class, 'account_id');
    }

    public function transactions(): HasMany
    {
        return $this->hasMany(BankTransaction::class);
    }

    public function reconciliations(): HasMany
    {
        return $this->hasMany(BankReconciliation::class);
    }

    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    public function billPayments(): HasMany
    {
        return $this->hasMany(BillPayment::class);
    }

    public function isConnectedToPlaid(): bool
    {
        return !empty($this->plaid_access_token);
    }

    public function updateBalance(): void
    {
        $credits = $this->transactions()
            ->where('type', 'credit')
            ->where('is_reconciled', true)
            ->sum('amount');

        $debits = $this->transactions()
            ->where('type', 'debit')
            ->where('is_reconciled', true)
            ->sum('amount');

        $this->current_balance = $this->opening_balance + $credits - $debits;
        $this->save();
    }

    public function getUnreconciledTransactions()
    {
        return $this->transactions()
            ->where('is_reconciled', false)
            ->orderBy('transaction_date', 'desc')
            ->get();
    }

    public function getLastReconciliation(): ?BankReconciliation
    {
        return $this->reconciliations()
            ->where('status', 'completed')
            ->latest('statement_date')
            ->first();
    }

    public static function getDefault($tenantId)
    {
        return static::where('tenant_id', $tenantId)
            ->where('is_default', true)
            ->where('is_active', true)
            ->first();
    }
}
