<?php

namespace App\Jobs;

use App\Models\ChartOfAccount;
use App\Models\Tenant;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Storage;

class GenerateBalanceSheet implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        protected Tenant $tenant,
        protected string $asOfDate,
        protected ?int $userId = null
    ) {}

    public function handle(): array
    {
        $accounts = ChartOfAccount::where('tenant_id', $this->tenant->id)
            ->with(['journalEntryLines' => function ($query) {
                $query->whereHas('journalEntry', function ($q) {
                    $q->where('date', '<=', $this->asOfDate);
                });
            }])
            ->get();

        $assets = $accounts->where('type', 'asset')->map(fn($a) => [
            'name' => $a->name,
            'balance' => $a->journalEntryLines->sum('debit') - $a->journalEntryLines->sum('credit'),
        ]);

        $liabilities = $accounts->where('type', 'liability')->map(fn($a) => [
            'name' => $a->name,
            'balance' => $a->journalEntryLines->sum('credit') - $a->journalEntryLines->sum('debit'),
        ]);

        $equity = $accounts->where('type', 'equity')->map(fn($a) => [
            'name' => $a->name,
            'balance' => $a->journalEntryLines->sum('credit') - $a->journalEntryLines->sum('debit'),
        ]);

        return [
            'as_of_date' => $this->asOfDate,
            'assets' => [
                'items' => $assets->values(),
                'total' => $assets->sum('balance'),
            ],
            'liabilities' => [
                'items' => $liabilities->values(),
                'total' => $liabilities->sum('balance'),
            ],
            'equity' => [
                'items' => $equity->values(),
                'total' => $equity->sum('balance'),
            ],
        ];
    }
}
