<?php

namespace App\Http\Controllers\Api;

use App\Models\WhatsappMessage;
use App\Models\Contact;
use App\Services\WhatsappService;
use Illuminate\Http\Request;

class WhatsappController extends ApiController
{
    protected WhatsappService $whatsappService;

    public function __construct(WhatsappService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    public function index(Request $request)
    {
        $query = WhatsappMessage::where('tenant_id', $request->user()->tenant_id)
            ->with('contact');

        if ($request->direction) {
            $query->where('direction', $request->direction);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->contact_id) {
            $query->where('contact_id', $request->contact_id);
        }

        if ($request->message_type) {
            $query->where('message_type', $request->message_type);
        }

        $messages = $query->latest()->paginate($request->per_page ?? 15);

        return $this->paginated($messages);
    }

    public function send(Request $request)
    {
        $validated = $request->validate([
            'to' => 'required|string',
            'message' => 'required|string|max:4096',
            'contact_id' => 'nullable|exists:contacts,id',
        ]);

        $tenantId = $request->user()->tenant_id;

        $whatsapp = WhatsappMessage::create([
            'tenant_id' => $tenantId,
            'contact_id' => $validated['contact_id'],
            'direction' => 'outbound',
            'from_number' => config('services.whatsapp.from_number'),
            'to_number' => $validated['to'],
            'message_type' => 'text',
            'content' => $validated['message'],
            'status' => 'queued',
        ]);

        try {
            $result = $this->whatsappService->sendTextMessage($validated['to'], $validated['message']);
            
            $whatsapp->update([
                'status' => 'sent',
                'sent_at' => now(),
                'provider_id' => $result['message_id'] ?? null,
                'provider_response' => $result,
            ]);
        } catch (\Exception $e) {
            $whatsapp->update([
                'status' => 'failed',
                'failed_at' => now(),
                'error_message' => $e->getMessage(),
            ]);

            return $this->error('Failed to send WhatsApp message: ' . $e->getMessage(), 500);
        }

        return $this->success($whatsapp, 'Message sent', 201);
    }

    public function sendTemplate(Request $request)
    {
        $validated = $request->validate([
            'to' => 'required|string',
            'template_name' => 'required|string',
            'template_params' => 'nullable|array',
            'contact_id' => 'nullable|exists:contacts,id',
        ]);

        $tenantId = $request->user()->tenant_id;

        $whatsapp = WhatsappMessage::create([
            'tenant_id' => $tenantId,
            'contact_id' => $validated['contact_id'],
            'direction' => 'outbound',
            'from_number' => config('services.whatsapp.from_number'),
            'to_number' => $validated['to'],
            'message_type' => 'template',
            'template_name' => $validated['template_name'],
            'template_params' => $validated['template_params'] ?? [],
            'status' => 'queued',
        ]);

        try {
            $result = $this->whatsappService->sendTemplate(
                $validated['to'],
                $validated['template_name'],
                $validated['template_params'] ?? []
            );
            
            $whatsapp->update([
                'status' => 'sent',
                'sent_at' => now(),
                'provider_id' => $result['message_id'] ?? null,
                'provider_response' => $result,
            ]);
        } catch (\Exception $e) {
            $whatsapp->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);

            return $this->error('Failed to send template: ' . $e->getMessage(), 500);
        }

        return $this->success($whatsapp, 'Template sent', 201);
    }

    public function sendMedia(Request $request)
    {
        $validated = $request->validate([
            'to' => 'required|string',
            'media' => 'required|file|max:16384',
            'caption' => 'nullable|string|max:1024',
            'contact_id' => 'nullable|exists:contacts,id',
        ]);

        $file = $request->file('media');
        $mimeType = $file->getMimeType();
        
        // Determine message type from mime
        $messageType = match (true) {
            str_starts_with($mimeType, 'image/') => 'image',
            str_starts_with($mimeType, 'video/') => 'video',
            str_starts_with($mimeType, 'audio/') => 'audio',
            default => 'document',
        };

        // Store file
        $path = $file->store('whatsapp-media', 'public');
        $mediaUrl = url('storage/' . $path);

        $tenantId = $request->user()->tenant_id;

        $whatsapp = WhatsappMessage::create([
            'tenant_id' => $tenantId,
            'contact_id' => $validated['contact_id'],
            'direction' => 'outbound',
            'from_number' => config('services.whatsapp.from_number'),
            'to_number' => $validated['to'],
            'message_type' => $messageType,
            'content' => $validated['caption'] ?? null,
            'media_url' => $mediaUrl,
            'media_mime_type' => $mimeType,
            'media_filename' => $file->getClientOriginalName(),
            'status' => 'queued',
        ]);

        try {
            $result = $this->whatsappService->sendMedia(
                $validated['to'],
                $mediaUrl,
                $messageType,
                $validated['caption'] ?? null
            );
            
            $whatsapp->update([
                'status' => 'sent',
                'sent_at' => now(),
                'provider_id' => $result['message_id'] ?? null,
            ]);
        } catch (\Exception $e) {
            $whatsapp->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);

            return $this->error('Failed to send media: ' . $e->getMessage(), 500);
        }

        return $this->success($whatsapp, 'Media sent', 201);
    }

    public function conversation(Contact $contact, Request $request)
    {
        $messages = WhatsappMessage::where('tenant_id', $request->user()->tenant_id)
            ->where('contact_id', $contact->id)
            ->orderBy('created_at')
            ->get();

        return $this->success([
            'contact' => $contact,
            'messages' => $messages,
            'whatsapp_link' => 'https://wa.me/' . preg_replace('/[^0-9]/', '', $contact->phone),
        ]);
    }

    public function reply(Contact $contact, Request $request)
    {
        $validated = $request->validate([
            'message' => 'required|string|max:4096',
        ]);

        return $this->send(new Request([
            'to' => $contact->whatsapp ?? $contact->phone,
            'message' => $validated['message'],
            'contact_id' => $contact->id,
        ]));
    }

    public function templates()
    {
        try {
            $templates = $this->whatsappService->getTemplates();
            return $this->success($templates);
        } catch (\Exception $e) {
            return $this->error('Failed to fetch templates: ' . $e->getMessage(), 500);
        }
    }

    /**
     * WhatsApp webhook for inbound messages
     */
    public function webhook(Request $request)
    {
        $payload = $request->all();

        // Handle incoming messages
        if (isset($payload['entry'][0]['changes'][0]['value']['messages'])) {
            foreach ($payload['entry'][0]['changes'][0]['value']['messages'] as $message) {
                $this->processInboundMessage($message, $payload);
            }
        }

        // Handle status updates
        if (isset($payload['entry'][0]['changes'][0]['value']['statuses'])) {
            foreach ($payload['entry'][0]['changes'][0]['value']['statuses'] as $status) {
                $this->processStatusUpdate($status);
            }
        }

        return response('OK', 200);
    }

    /**
     * WhatsApp webhook verification
     */
    public function webhookVerify(Request $request)
    {
        $verifyToken = config('services.whatsapp.verify_token');

        if ($request->input('hub_verify_token') === $verifyToken) {
            return response($request->input('hub_challenge'));
        }

        return response('Forbidden', 403);
    }

    protected function processInboundMessage(array $message, array $payload)
    {
        $from = $message['from'];
        $messageType = $message['type'];
        $timestamp = $message['timestamp'];

        $contact = Contact::where('phone', $from)
            ->orWhere('whatsapp', $from)
            ->first();

        $tenantId = $contact?->tenant_id ?? config('app.default_tenant_id');

        $content = match ($messageType) {
            'text' => $message['text']['body'] ?? null,
            'image', 'document', 'audio', 'video' => $message[$messageType]['caption'] ?? null,
            default => null,
        };

        WhatsappMessage::create([
            'tenant_id' => $tenantId,
            'contact_id' => $contact?->id,
            'direction' => 'inbound',
            'from_number' => $from,
            'to_number' => $payload['entry'][0]['changes'][0]['value']['metadata']['display_phone_number'] ?? null,
            'message_type' => $messageType,
            'content' => $content,
            'media_url' => $message[$messageType]['id'] ?? null,
            'status' => 'received',
            'provider_id' => $message['id'],
        ]);
    }

    protected function processStatusUpdate(array $status)
    {
        $messageId = $status['id'];
        $statusValue = $status['status'];

        $whatsapp = WhatsappMessage::where('provider_id', $messageId)->first();

        if ($whatsapp) {
            $updateData = match ($statusValue) {
                'delivered' => ['status' => 'delivered', 'delivered_at' => now()],
                'read' => ['status' => 'read', 'read_at' => now()],
                'failed' => ['status' => 'failed', 'failed_at' => now(), 'error_message' => $status['errors'][0]['message'] ?? 'Unknown error'],
                default => [],
            };

            if ($updateData) {
                $whatsapp->update($updateData);
            }
        }
    }
}
