<?php

namespace App\Http\Controllers\Api;

use App\Models\Tenant;
use Illuminate\Http\Request;

class TenantController extends ApiController
{
    public function show(Request $request)
    {
        return $this->success($request->user()->tenant);
    }

    public function update(Request $request)
    {
        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'website' => 'nullable|url|max:255',
            'address' => 'nullable|array',
            'tax_number' => 'nullable|string|max:50',
            'vat_number' => 'nullable|string|max:50',
            'settings' => 'nullable|array',
        ]);

        $tenant = $request->user()->tenant;
        $tenant->update($validated);

        return $this->success($tenant, 'Tenant updated successfully');
    }

    public function updateBranding(Request $request)
    {
        $validated = $request->validate([
            'logo' => 'nullable|image|max:2048',
            'favicon' => 'nullable|image|max:512',
            'primary_color' => 'nullable|string|max:7',
            'secondary_color' => 'nullable|string|max:7',
        ]);

        $tenant = $request->user()->tenant;

        if ($request->hasFile('logo')) {
            $path = $request->file('logo')->store("tenants/{$tenant->id}/branding", 'public');
            $tenant->logo = $path;
        }

        if ($request->hasFile('favicon')) {
            $path = $request->file('favicon')->store("tenants/{$tenant->id}/branding", 'public');
            $tenant->favicon = $path;
        }

        if (isset($validated['primary_color'])) {
            $tenant->settings = array_merge($tenant->settings ?? [], [
                'primary_color' => $validated['primary_color'],
            ]);
        }

        $tenant->save();

        return $this->success($tenant, 'Branding updated successfully');
    }

    public function billing(Request $request)
    {
        $tenant = $request->user()->tenant;

        return $this->success([
            'subscription' => $tenant->subscription ?? null,
            'plan' => $tenant->plan ?? 'free',
            'billing_email' => $tenant->billing_email,
            'next_billing_date' => $tenant->next_billing_date,
        ]);
    }

    public function usage(Request $request)
    {
        $tenant = $request->user()->tenant;

        return $this->success([
            'users' => $tenant->users()->count(),
            'contacts' => $tenant->contacts()->count(),
            'invoices' => $tenant->invoices()->count(),
            'storage_used' => $this->calculateStorageUsed($tenant),
        ]);
    }

    protected function calculateStorageUsed(Tenant $tenant): int
    {
        // Calculate storage in bytes
        return 0; // Implement based on your storage tracking
    }

    public function destroy(Request $request)
    {
        $request->validate([
            'confirmation' => 'required|in:DELETE',
            'password' => 'required',
        ]);

        if (!\Hash::check($request->password, $request->user()->password)) {
            return $this->error('Invalid password', 403);
        }

        $tenant = $request->user()->tenant;
        
        // Soft delete tenant and all related data
        $tenant->delete();

        return $this->success(null, 'Tenant deleted successfully');
    }
}
