<?php

namespace App\Http\Controllers\Api;

use App\Models\Tag;
use Illuminate\Http\Request;

class TagController extends ApiController
{
    public function index(Request $request)
    {
        $query = Tag::where('tenant_id', $request->user()->tenant_id)
            ->withCount('contacts');

        if ($request->search) {
            $query->where('name', 'like', "%{$request->search}%");
        }

        $tags = $query->orderBy('name')->get();

        return $this->success($tags);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:50',
            'color' => 'nullable|string|max:7',
            'description' => 'nullable|string|max:255',
        ]);

        $tag = Tag::create([
            'tenant_id' => $request->user()->tenant_id,
            ...$validated,
        ]);

        return $this->success($tag, 'Tag created', 201);
    }

    public function show(Tag $tag)
    {
        $this->authorize('view', $tag);

        return $this->success($tag->loadCount('contacts'));
    }

    public function update(Request $request, Tag $tag)
    {
        $this->authorize('update', $tag);

        $validated = $request->validate([
            'name' => 'sometimes|string|max:50',
            'color' => 'nullable|string|max:7',
            'description' => 'nullable|string|max:255',
        ]);

        $tag->update($validated);

        return $this->success($tag, 'Tag updated');
    }

    public function destroy(Tag $tag)
    {
        $this->authorize('delete', $tag);

        // Detach from all contacts first
        $tag->contacts()->detach();
        $tag->delete();

        return $this->success(null, 'Tag deleted');
    }

    public function contacts(Tag $tag, Request $request)
    {
        $this->authorize('view', $tag);

        $contacts = $tag->contacts()
            ->with('owner')
            ->paginate($request->per_page ?? 15);

        return $this->paginated($contacts);
    }

    public function merge(Request $request)
    {
        $validated = $request->validate([
            'source_ids' => 'required|array|min:1',
            'source_ids.*' => 'exists:tags,id',
            'target_id' => 'required|exists:tags,id',
        ]);

        $targetTag = Tag::findOrFail($validated['target_id']);
        $sourceTags = Tag::whereIn('id', $validated['source_ids'])->get();

        foreach ($sourceTags as $sourceTag) {
            if ($sourceTag->id === $targetTag->id) continue;

            // Move contacts to target tag
            foreach ($sourceTag->contacts as $contact) {
                if (!$targetTag->contacts()->where('contact_id', $contact->id)->exists()) {
                    $targetTag->contacts()->attach($contact->id);
                }
            }

            $sourceTag->contacts()->detach();
            $sourceTag->delete();
        }

        return $this->success($targetTag->loadCount('contacts'), 'Tags merged successfully');
    }
}
